<!DOCTYPE html>
<html lang="en">
<head>
    <title>Upgrade Your Plan</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <style>
        .loading {
            display: none;
            width: 20px;
            height: 20px;
            border: 3px solid #f3f3f3;
            border-top: 3px solid #3498db;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin-left: 10px;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
    <script src="https://checkout.razorpay.com/v1/checkout.js"></script>
</head>
<body class="bg-gray-100 font-sans">
    <div class="container mx-auto p-6">
        <h1 class="text-3xl font-bold text-gray-800 mb-6">Upgrade Your Plan</h1>
        @if (session('error'))
            <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded mb-4" role="alert">
                <p>{{ session('error') }}</p>
            </div>
        @endif
        <div id="upgradeForm" class="bg-white p-6 rounded-lg shadow-lg">
            <div class="mb-4">
                <label for="planSelect" class="block text-gray-700 text-sm font-bold mb-2">Select a Plan</label>
                <select id="planSelect" class="w-full p-3 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500">
                    <option value="">Choose a plan to upgrade</option>
                    @foreach ($plans as $plan)
                        <option value="{{ $plan->Plan_id }}" data-price="{{ $plan->Price }}" data-currency="{{ $plan->currency->Currency_code }}">
                            {{ $plan->Name }} ({{ $plan->Price }} {{ $plan->currency->Currency_code }})
                        </option>
                    @endforeach
                </select>
            </div>
            <input type="hidden" id="subscriptionId" value="{{ $subscription->Subscription_id }}">
            <input type="hidden" id="currentUrl" value="{{ url()->current() }}">
            <button id="upgradeButton" class="bg-blue-600 text-white px-4 py-2 rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 disabled:bg-gray-400 disabled:cursor-not-allowed flex items-center" disabled>
                Upgrade Now
                <span id="loadingSpinner" class="loading ml-2"></span>
            </button>
            <div id="result" class="mt-4 text-sm"></div>
        </div>
    </div>

    <script>
        const planSelect = document.getElementById('planSelect');
        const upgradeButton = document.getElementById('upgradeButton');
        const loadingSpinner = document.getElementById('loadingSpinner');
        const resultDiv = document.getElementById('result');
        const subscriptionId = document.getElementById('subscriptionId').value;
        const currentUrl = document.getElementById('currentUrl').value;

        planSelect.addEventListener('change', () => {
            upgradeButton.disabled = !planSelect.value;
        });

        upgradeButton.addEventListener('click', () => {
            const planId = planSelect.value;
            const price = planSelect.options[planSelect.selectedIndex].getAttribute('data-price');
            const currency = planSelect.options[planSelect.selectedIndex].getAttribute('data-currency');

            if (!planId) {
                resultDiv.innerHTML = '<p class="text-red-500">Please select a plan.</p>';
                return;
            }

            upgradeButton.disabled = true;
            loadingSpinner.style.display = 'inline-block';

            fetch('/api/upgrade/plan', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                },
                body: JSON.stringify({
                    Subscription_id: subscriptionId,
                    New_Plan_id: planId,
                    Amount: price,
                    Payment_gateway: 'razorpay',
                    Currency: currency
                })
            })
            .then(response => response.json())
            .then(data => {
                loadingSpinner.style.display = 'none';
                if (data.status === 'pending') {
                    resultDiv.innerHTML = `<p class="text-yellow-500">${data.message}</p>`;
                    const options = {
                        key: data.data.key,
                        amount: data.data.amount * 100,
                        currency: data.data.currency,
                        name: data.data.name,
                        description: data.data.description,
                        order_id: data.data.order_id,
                        handler: function (response) {
                            fetch('/api/upgrade/callback/razorpay', {
                                method: 'POST',
                                headers: {
                                    'Content-Type': 'application/json',
                                    'X-CSRF-TOKEN': '{{ csrf_token() }}'
                                },
                                body: JSON.stringify({
                                    Transaction_id: data.data.order_id,
                                    razorpay_payment_id: response.razorpay_payment_id,
                                    razorpay_order_id: response.razorpay_order_id,
                                    razorpay_signature: response.razorpay_signature,
                                    New_Plan_id: planId
                                })
                            })
                            .then(cbResponse => cbResponse.json())
                            .then(cbData => {
                                if (cbData.status === 'success') {
                                    resultDiv.innerHTML = `<p class="text-green-500">${cbData.message}</p>`;
                                    setTimeout(() => {
                                        window.location.href = currentUrl;
                                    }, 2000);
                                } else {
                                    resultDiv.innerHTML = `<p class="text-red-500">Error: ${cbData.message}</p>`;
                                }
                            })
                            .catch(cbError => {
                                resultDiv.innerHTML = `<p class="text-red-500">Error: ${cbError.message}</p>`;
                            });
                        },
                        prefill: {
                            name: '{{ $subscription->company->Company_name }}',
                            email: '{{ $subscription->company->user->email ?? '' }}',
                            contact: '{{ $subscription->company->user->phone ?? '' }}'
                        },
                        notes: {
                            subscription_id: subscriptionId,
                            new_plan_id: planId
                        },
                        theme: {
                            color: '#3399cc'
                        }
                    };
                    const rzp = new Razorpay(options);
                    rzp.open();
                    rzp.on('payment.failed', function (response) {
                        resultDiv.innerHTML = `<p class="text-red-500">Payment failed: ${response.error.description}</p>`;
                    });
                } else if (data.status === 'success') {
                    resultDiv.innerHTML = `<p class="text-green-500">${data.message}</p>`;
                    setTimeout(() => {
                        window.location.href = currentUrl;
                    }, 2000);
                } else {
                    resultDiv.innerHTML = `<p class="text-red-500">Error: ${data.message || 'Upgrade failed'}</p>`;
                }
            })
            .catch(error => {
                loadingSpinner.style.display = 'none';
                resultDiv.innerHTML = `<p class="text-red-500">Error: ${error.message}</p>`;
            })
            .finally(() => {
                upgradeButton.disabled = false;
            });
        });
    </script>
</body>
</html>
