<?php

namespace App\Listeners;

use App\Events\InterviewStatusChanged;
use App\Models\Notification;
use Illuminate\Support\Facades\Log;

class SendInterviewStatusNotification
{
    public function handle(InterviewStatusChanged $event)
    {
        try {
            $interview = $event->interview;
            $application = $interview->application;
            $job = $application->job;

            // Define notification message based on status
            $statusMessages = [
                'scheduled' => 'Your interview for the job "' . ($job->title ?? 'Untitled Job') . '" has been scheduled on ' . $interview->Date->toDateTimeString() . '.',
                'completed' => 'Your interview for the job "' . ($job->title ?? 'Untitled Job') . '" was completed on ' . $interview->Date->toDateTimeString() . '.',
                'canceled' => 'Your interview for the job "' . ($job->title ?? 'Untitled Job') . '" has been canceled.',
            ];
            $message = $statusMessages[$interview->Status] ?? 'Your interview status for job "' . ($job->title ?? 'Untitled Job') . '" has been updated.';

            // Notify only the candidate who applied
            if ($application && $application->user && $application->user->User_type === 'candidate') {
                $notification = Notification::create([
                    'User_id' => $application->User_id,
                    'Title' => 'Interview Status Update',
                    'Message' => $message,
                    'Type' => 'info',
                    'Is_read' => false,
                    'Created_at' => now(),
                    'Updated_at' => now(),
                ]);

                Log::info('Notification created for interview status to candidate', [
                    'notification_id' => $notification->Notification_id,
                    'user_id' => $application->User_id,
                    'interview_id' => $interview->Interview_id,
                    'status' => $interview->Status,
                ]);
            } else {
                Log::warning('No valid candidate user found for interview notification', [
                    'interview_id' => $interview->Interview_id,
                    'application_id' => $application ? $application->Application_id : null,
                    'user_type' => $application && $application->user ? $application->user->User_type : null,
                ]);
            }
        } catch (\Exception $e) {
            Log::error('Error sending interview status notification: ' . $e->getMessage(), [
                'interview_id' => $event->interview->Interview_id,
            ]);
        }
    }
}
