<?php

namespace App\Http\Controllers;

use App\Models\VideoProgress;
use App\Models\Lesson;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class VideoProgressController extends Controller
{
    public function index(): JsonResponse
    {
        $progress = VideoProgress::with('lesson')->get();
        return response()->json([
            'status' => 'success',
            'data' => $progress,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $progress = VideoProgress::with('lesson')->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $progress,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:users,User_id',
            'Lesson_id' => 'required|exists:lessons,Lesson_id',
            'Watched_duration' => 'required|integer|min:0',
            'Last_position' => 'sometimes|integer|min:0',
            'Is_completed' => 'sometimes|boolean',
            'Interactive_elements_completed' => 'sometimes|integer|min:0',
            'Assignment_id' => 'sometimes|nullable|exists:assignments,Assignment_id',
            'Last_activity' => 'sometimes|date',
            'Total_attempts' => 'sometimes|integer|min:1',
            'Completion_status' => 'sometimes|in:in_progress,completed,failed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'User_id',
            'Lesson_id',
            'Watched_duration',
            'Last_position',
            'Is_completed',
            'Interactive_elements_completed',
            'Assignment_id',
            'Last_activity',
            'Total_attempts',
            'Completion_status',
        ]);

        // Fetch duration from lesson and convert to seconds
        $lesson = Lesson::findOrFail($data['Lesson_id']);
        if ($lesson->Video_link && !$lesson->Duration) {
            $lessonDurationMinutes = $this->getYouTubeDuration($lesson->Video_link); // Returns minutes
            $lesson->update(['Duration' => $lessonDurationMinutes]);
        }
        $data['Total_duration'] = ($lesson->Duration ? $lesson->Duration * 60 : 300); // Convert to seconds, 300s fallback

        // Auto-set Is_completed if Last_position reaches or exceeds Total_duration
        if (!isset($data['Is_completed'])) {
            $lastPosition = $data['Last_position'] ?? 0;
            $data['Is_completed'] = $lastPosition >= $data['Total_duration'];
            if ($data['Is_completed']) {
                $data['Completed_at'] = now();
                $data['Completion_status'] = 'completed';
            } else {
                $data['Completion_status'] = $data['Watched_duration'] > 0 ? 'in_progress' : 'in_progress';
            }
        }

        $data['Last_activity'] = $data['Last_activity'] ?? now();
        $existing = VideoProgress::where('User_id', $data['User_id'])->where('Lesson_id', $data['Lesson_id'])->first();
        $data['Total_attempts'] = $existing ? $existing->Total_attempts + 1 : ($data['Total_attempts'] ?? 1);

        $progress = VideoProgress::create($data);

        return response()->json([
            'status' => 'success',
            'data' => $progress,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $progress = VideoProgress::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Watched_duration' => 'sometimes|integer|min:0',
            'Last_position' => 'sometimes|integer|min:0',
            'Is_completed' => 'sometimes|boolean',
            'Interactive_elements_completed' => 'sometimes|integer|min:0',
            'Assignment_id' => 'sometimes|nullable|exists:assignments,Assignment_id',
            'Last_activity' => 'sometimes|date',
            'Total_attempts' => 'sometimes|integer|min:1',
            'Completion_status' => 'sometimes|in:in_progress,completed,failed',
            'Total_duration' => 'sometimes|integer|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'Watched_duration',
            'Last_position',
            'Is_completed',
            'Interactive_elements_completed',
            'Assignment_id',
            'Last_activity',
            'Total_attempts',
            'Completion_status',
            'Total_duration',
        ]);

        // Fetch duration from lesson or existing record if not provided
        $lesson = Lesson::findOrFail($progress->Lesson_id);
        if (!isset($data['Total_duration'])) {
            if ($lesson->Video_link && !$progress->Total_duration) {
                $lessonDurationMinutes = $this->getYouTubeDuration($lesson->Video_link); // Returns minutes
                $data['Total_duration'] = $lessonDurationMinutes * 60; // To seconds
                $lesson->update(['Duration' => $lessonDurationMinutes]);
            } elseif ($progress->Total_duration) {
                $data['Total_duration'] = $progress->Total_duration;
            } else {
                $data['Total_duration'] = ($lesson->Duration ? $lesson->Duration * 60 : 300); // Convert to seconds
            }
        }

        // Auto-set Is_completed only if Last_position reaches or exceeds Total_duration
        $originalIsCompleted = $progress->Is_completed;
        if (isset($data['Last_position']) && !isset($data['Is_completed'])) {
            $data['Is_completed'] = $data['Last_position'] >= $data['Total_duration'];
            if ($data['Is_completed'] && !$originalIsCompleted) {
                $data['Completed_at'] = now();
                $data['Completion_status'] = 'completed';
                Log::info("Lesson completed for User {$progress->User_id}, Lesson {$progress->Lesson_id}, Progress_id {$id}");
            } elseif (!$data['Is_completed']) {
                $data['Completion_status'] = $data['Watched_duration'] > 0 ? 'in_progress' : $progress->Completion_status;
                $data['Completed_at'] = null;
            }
        } elseif (!isset($data['Completion_status'])) {
            $data['Completion_status'] = $data['Watched_duration'] > 0 ? 'in_progress' : $progress->Completion_status;
        }

        // Update Last_activity if provided or set to now
        if (!isset($data['Last_activity'])) {
            $data['Last_activity'] = now();
        }

        // Increment Total_attempts if updated
        if (isset($data['Total_attempts'])) {
            $data['Total_attempts'] = $progress->Total_attempts + 1;
        }

        Log::info('Setting Total_duration to: ' . $data['Total_duration']); // Debug log
        $progress->update($data);

        return response()->json([
            'status' => 'success',
            'data' => $progress,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $progress = VideoProgress::findOrFail($id);
        $progress->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Video progress deleted successfully',
        ], 200);
    }

    /**
     * Fetch YouTube video duration
     */
    private function getYouTubeDuration($videoUrl)
    {
        if (!$videoUrl) return 5; // Fallback in minutes
        preg_match('/v=([^&]+)/', $videoUrl, $matches);
        $videoId = $matches[1] ?? '';
        $apiKey = env('YOUTUBE_API_KEY', 'AIzaSyDiAiyNJJEnAX7y9zX8zCu-WsPj_f1Z-Jw');
        $url = "https://www.googleapis.com/youtube/v3/videos?part=contentDetails&id={$videoId}&key={$apiKey}";
        $response = @file_get_contents($url);
        if ($response === false) {
            Log::warning("Failed to fetch YouTube duration for video ID: {$videoId}");
            return 5; // Fallback in minutes
        }
        $data = json_decode($response, true);
        if (isset($data['items'][0]['contentDetails']['duration'])) {
            $isoDuration = $data['items'][0]['contentDetails']['duration'];
            return round($this->iso8601ToSeconds($isoDuration) / 60); // Return minutes, rounded
        }
        return 5; // Fallback
    }

    private function iso8601ToSeconds($duration)
    {
        $interval = new \DateInterval($duration);
        return ($interval->d * 86400) + ($interval->h * 3600) + ($interval->i * 60) + $interval->s;
    }
}
