<?php

namespace App\Http\Controllers;

use App\Models\UserGroup;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class UserGroupController extends Controller
{
    public function index(): JsonResponse
    {
        $userGroups = UserGroup::with(['user', 'group'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $userGroups,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $userGroup = UserGroup::with(['user', 'group'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $userGroup,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:Users,User_id',
            'Group_id' => 'required|exists:Groups,Group_id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $userGroup = UserGroup::create(array_merge(
            $request->only([
                'User_id',
                'Group_id',
            ]),
            ['Joined_at' => now()]
        ));

        return response()->json([
            'status' => 'success',
            'data' => $userGroup,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $userGroup = UserGroup::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'User_id' => 'sometimes|exists:Users,User_id',
            'Group_id' => 'sometimes|exists:Groups,Group_id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $userGroup->update($request->only([
            'User_id',
            'Group_id',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $userGroup,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $userGroup = UserGroup::findOrFail($id);
        $userGroup->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'User group membership deleted successfully',
        ], 200);
    }
}
