<?php

namespace App\Http\Controllers;

use App\Models\UserCourse;
use App\Models\Course;
use App\Models\Payment;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class UserCourseController extends Controller
{
    public function index(): JsonResponse
    {
        $userCourses = UserCourse::with(['user', 'course', 'payment'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $userCourses,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $userCourse = UserCourse::with(['user', 'course', 'payment'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $userCourse,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:users,User_id',
            'Course_id' => 'required|exists:courses,Course_id',
            'Progress' => 'sometimes|numeric|min:0|max:100',
            'Completed_at' => 'nullable|date',
            'Status' => 'sometimes|in:pending,enrolled,in_progress,completed,dropped',
            'Payment_id' => 'nullable|exists:payments,Payment_id',
            'Assignments_completed' => 'sometimes|integer|min:0',
            'Practice_tests_completed' => 'sometimes|integer|min:0',
            'Mock_interviews_completed' => 'sometimes|integer|min:0',
            'Last_activity' => 'sometimes|date',
            'Total_attempts' => 'sometimes|integer|min:1',
            'Completion_status' => 'sometimes|in:in_progress,completed,failed',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()], 422);
        }

        $existingEnrollment = UserCourse::where('User_id', $request->User_id)
            ->where('Course_id', $request->Course_id)
            ->first();
        if ($existingEnrollment) {
            return response()->json(['status' => 'error', 'message' => 'User is already enrolled in this course'], 409);
        }

        $course = Course::find($request->Course_id);
        if ($course->Is_paid) {
            return response()->json(['status' => 'error', 'message' => 'Enrollment for paid courses is handled after payment completion'], 400);
        }

        $userCourse = DB::transaction(function () use ($request) {
            $userCourseData = array_merge(
                $request->only([
                    'User_id',
                    'Course_id',
                    'Progress',
                    'Completed_at',
                    'Payment_id',
                    'Assignments_completed',
                    'Practice_tests_completed',
                    'Mock_interviews_completed',
                    'Last_activity',
                    'Total_attempts',
                    'Completion_status',
                ]),
                [
                    'Enrollment_number' => Str::uuid()->toString(),
                    'Status' => 'enrolled',
                    'created_at' => now(),
                    'updated_at' => now(),
                    'Last_activity' => $request->Last_activity ?? now(), // Default to now()
                    'Total_attempts' => $request->Total_attempts ?? 1, // Default to 1
                    'Completion_status' => $request->Completion_status ?? 'in_progress', // Default
                ]
            );

            return UserCourse::create($userCourseData);
        });

        return response()->json(['status' => 'success', 'data' => $userCourse], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $userCourse = UserCourse::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'User_id' => 'sometimes|exists:users,User_id',
            'Course_id' => 'sometimes|exists:courses,Course_id',
            'Progress' => 'sometimes|numeric|min:0|max:100',
            'Completed_at' => 'nullable|date',
            'Status' => 'sometimes|in:pending,enrolled,in_progress,completed,dropped',
            'Payment_id' => 'nullable|exists:payments,Payment_id',
            'Assignments_completed' => 'sometimes|integer|min:0',
            'Practice_tests_completed' => 'sometimes|integer|min:0',
            'Mock_interviews_completed' => 'sometimes|integer|min:0',
            'Last_activity' => 'sometimes|date',
            'Total_attempts' => 'sometimes|integer|min:1',
            'Completion_status' => 'sometimes|in:in_progress,completed,failed',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $userCourse = DB::transaction(function () use ($request, $userCourse) {
            $userCourse->update(array_merge(
                $request->only([
                    'User_id',
                    'Course_id',
                    'Progress',
                    'Completed_at',
                    'Status',
                    'Payment_id',
                    'Assignments_completed',
                    'Practice_tests_completed',
                    'Mock_interviews_completed',
                    'Last_activity',
                    'Total_attempts',
                    'Completion_status',
                ]),
                ['updated_at' => now()]
            ));

            // Update Last_activity if not provided
            if (!isset($request->Last_activity)) {
                $userCourse->Last_activity = now();
            }

            // Increment Total_attempts
            $userCourse->Total_attempts = $userCourse->Total_attempts + 1;

            // Sync Completion_status with Progress and Status
            if (isset($request->Progress) && $request->Progress >= 100) {
                $userCourse->Completion_status = 'completed';
                $userCourse->Status = 'completed';
                $userCourse->Completed_at = now();
            } elseif (isset($request->Progress) && $request->Progress > 0) {
                $userCourse->Completion_status = 'in_progress';
                $userCourse->Status = 'in_progress';
            } elseif (!isset($request->Completion_status)) {
                $userCourse->Completion_status = $userCourse->Progress > 0 ? 'in_progress' : 'enrolled';
            }

            $userCourse->save();

            return $userCourse;
        });

        return response()->json([
            'status' => 'success',
            'data' => $userCourse,
            'message' => 'User course updated successfully.',
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $userCourse = UserCourse::findOrFail($id);
        $userCourse->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'User course enrollment deleted successfully',
        ], 200);
    }

    public function approve($id): JsonResponse
    {
        $userCourse = UserCourse::findOrFail($id);
        if ($userCourse->Status !== 'pending') {
            return response()->json([
                'status' => 'error',
                'message' => 'Enrollment is not pending',
            ], 400);
        }

        $userCourse->update([
            'Status' => 'enrolled',
            'Last_activity' => now(), // Update on approval
            'Total_attempts' => $userCourse->Total_attempts + 1, // Increment on approval
            'Completion_status' => 'in_progress', // Default on approval
        ]);

        return response()->json([
            'status' => 'success',
            'data' => $userCourse,
            'message' => 'Enrollment approved successfully.',
        ], 200);
    }
}
