<?php

namespace App\Http\Controllers;

use App\Models\Subscription;
use App\Models\Plan;
use App\Models\Transaction;
use App\Models\Payment;
use App\Models\SubscriptionHistory;
use App\Services\PaymentHandler;
use App\Http\Controllers\CompanyFeatureUsageController;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class UpgradeController extends Controller
{
    public function showUpgradeForm($companyId)
    {
        $subscription = Subscription::where('Company_id', $companyId)->first();
        if (!$subscription) {
            return redirect()->back()->with('error', 'No subscription found');
        }

        $plans = Plan::where('Status', 'active')->where('Access_Level', '!=', 'basic')->get();
        return view('upgrade-plan', compact('subscription', 'plans'));
    }

    public function upgradePlan(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Subscription_id' => 'required|exists:subscriptions,Subscription_id',
            'New_Plan_id' => 'required|exists:plans,Plan_id',
            'Amount' => 'required|numeric|min:1',
            'Payment_gateway' => 'required|in:paypal,stripe,razorpay',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()], 422);
        }

        $subscription = Subscription::find($request->Subscription_id);
        if (!$subscription) {
            return response()->json(['status' => 'error', 'message' => 'Subscription not found'], 404);
        }

        $newPlan = Plan::find($request->New_Plan_id);
        if (!$newPlan) {
            return response()->json(['status' => 'error', 'message' => 'New plan not found'], 404);
        }

        return DB::transaction(function () use ($request, $subscription, $newPlan) {
            $gatewayAmount = round($request->Amount * 100);

            $payment = Payment::create([
                'User_id' => $subscription->company->User_id,
                'Subscription_id' => $subscription->Subscription_id,
                'Amount' => $request->Amount,
                'Currency' => 'INR',
                'Payment_gateway' => $request->Payment_gateway,
                'Payment_status' => 'pending',
            ]);

            try {
                $paymentHandler = new PaymentHandler($request->Payment_gateway);
                $response = $paymentHandler->initiatePayment($payment, [
                    'amount' => $gatewayAmount,
                    'currency' => 'INR',
                    'receipt' => 'receipt_' . $payment->Payment_id,
                ]);

                $payment->update(['Transaction_id' => $response['order_id'] ?? null]);

                if ($request->Payment_gateway === 'razorpay') {
                    return response()->json([
                        'status' => 'pending',
                        'message' => 'Razorpay payment initiated',
                        'data' => [
                            'order_id' => $response['order_id'],
                            'amount' => $request->Amount,
                            'currency' => 'INR',
                            'key' => config('services.razorpay.key'),
                            'name' => $subscription->company->Company_name,
                            'description' => 'Plan Upgrade to ' . $newPlan->Name,
                        ],
                    ], 200);
                }

                $subscription->update(['Plan_id' => $request->New_Plan_id]);
                return response()->json([
                    'status' => 'success',
                    'message' => 'Subscription upgraded successfully',
                    'data' => $subscription->load('plan'),
                ], 200);
            } catch (\Exception $e) {
                Log::error('Payment initiation failed', ['error' => $e->getMessage()]);
                $payment->update(['Payment_status' => 'failed', 'Transaction_id' => null]);
                return response()->json(['status' => 'error', 'message' => 'Payment initiation failed: ' . $e->getMessage()], 500);
            }
        });
    }

    public function callback(Request $request, $gateway): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Transaction_id' => 'required|string',
            'New_Plan_id' => 'required|exists:plans,Plan_id',
        ]);
        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()], 422);
        }

        return DB::transaction(function () use ($request, $gateway) {
            $paymentHandler = new PaymentHandler($gateway);
            $gatewayData = $request->only(['razorpay_payment_id', 'razorpay_order_id', 'razorpay_signature']);
            $verified = $paymentHandler->verifyPayment($request->Transaction_id, $gatewayData);

            if (!$verified) {
                Log::warning('Payment verification failed for Transaction_id: ' . $request->Transaction_id);
                return response()->json(['status' => 'error', 'message' => 'Payment verification failed'], 400);
            }

            $payment = Payment::where('Transaction_id', $request->Transaction_id)->lockForUpdate()->firstOrFail();
            $payment->update(['Payment_status' => 'completed']);

            // Check for existing transaction to prevent duplicates and ensure only 'Upgrade Plan' type
            $transaction = Transaction::where('Payment_id', $payment->Payment_id)->first();

            if ($transaction) {
                // Update existing transaction to 'Upgrade Plan' if it exists
                $transaction->update([
                    'Transaction_type' => 'Upgrade Plan',
                    'Status' => 'completed',
                    'Gateway_response' => json_encode($gatewayData),
                    'Processed_at' => now(),
                ]);
            } else {
                // Create new transaction only if none exists, with 'Upgrade Plan' type
                $transaction = Transaction::create([
                    'Payment_id' => $payment->Payment_id,
                    'Transaction_type' => 'Upgrade Plan',
                    'Amount' => $payment->Amount,
                    'Currency' => $payment->Currency,
                    'Payment_method' => $payment->Payment_gateway,
                    'Status' => 'completed',
                    'Gateway_response' => json_encode($gatewayData),
                    'Processed_at' => now(),
                ]);
            }

            $subscription = Subscription::find($payment->Subscription_id);
            $newPlanId = $request->input('New_Plan_id');
            $newPlan = Plan::find($newPlanId);

            // Update the subscription with the new plan
            $subscription->update(['Plan_id' => $newPlanId]);

            // Create a new SubscriptionHistory entry
            SubscriptionHistory::create([
                'subscription_id' => $subscription->Subscription_id,
                'plan_id' => $newPlanId,
                'company_id' => $subscription->Company_id,
                'name' => 'Subscription to ' . $newPlan->Name,
                'start_date' => now()->toDateTimeString(),
                'end_date' => null,
                'status' => 'active',
                'amount' => $payment->Amount,
                'payment_gateway' => $payment->Payment_gateway,
            ]);

            // Sync usage limits with the new plan and reset unused features
            $companyFeatureUsageController = new CompanyFeatureUsageController();
            $companyFeatureUsageController->syncUsageLimits($subscription->company->Company_id);

            return response()->json([
                'status' => 'success',
                'message' => 'Payment successfully completed and plan upgraded!',
                'data' => $subscription->load('plan'),
            ], 200);
        }, 5);
    }
}
