<?php

namespace App\Http\Controllers;

use App\Models\Subscription;
use App\Models\Plan;
use App\Models\SubscriptionHistory;
use App\Http\Controllers\PaymentController;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;

class SubscriptionController extends Controller
{
    public function index(): JsonResponse
    {
        $subscriptions = Subscription::all();
        return response()->json($subscriptions);
    }

    public function show($id): JsonResponse
    {
        $subscription = Subscription::find($id);
        if (!$subscription) {
            return response()->json(['message' => 'Subscription not found'], 404);
        }
        return response()->json($subscription);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Company_id' => 'required|integer|exists:company_profiles,Company_id',
            'Plan_id' => 'required|integer|exists:plans,Plan_id',
            'Start_date' => 'required|date',
            'End_date' => 'required|date|after:Start_date',
            'Status' => 'required|string|in:active,inactive,expired',
            'Name' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        // Create the subscription (no history logging for initial/default subscriptions)
        $subscription = Subscription::create($request->all());

        return response()->json($subscription, 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $subscription = Subscription::find($id);
        if (!$subscription) {
            return response()->json(['message' => 'Subscription not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'Company_id' => 'sometimes|integer|exists:company_profiles,Company_id',
            'Plan_id' => 'sometimes|integer|exists:plans,Plan_id',
            'Start_date' => 'sometimes|date',
            'End_date' => 'sometimes|date|after:Start_date',
            'Status' => 'sometimes|string|in:active,inactive,expired',
            'Name' => 'sometimes|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $subscription->update($request->all());

        return response()->json(['message' => 'Subscription updated successfully', 'data' => $subscription], 200);
    }

    public function destroy($id): JsonResponse
    {
        $subscription = Subscription::find($id);
        if (!$subscription) {
            return response()->json(['message' => 'Subscription not found'], 404);
        }

        $subscription->delete();
        return response()->json(['message' => 'Subscription deleted successfully'], 200);
    }

    public function upgrade(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Subscription_id' => 'required|exists:subscriptions,Subscription_id',
            'New_Plan_id' => 'required|exists:plans,Plan_id',
            'Amount' => 'required|numeric|min:1',
            'Payment_gateway' => 'required|in:paypal,stripe,razorpay',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()], 422);
        }

        $subscription = Subscription::with('company')->find($request->Subscription_id);
        if (!$subscription) {
            Log::error('Subscription not found during upgrade', ['subscription_id' => $request->Subscription_id]);
            return response()->json(['status' => 'error', 'message' => 'Subscription not found'], 404);
        }

        $newPlan = Plan::find($request->New_Plan_id);
        if (!$newPlan) {
            Log::error('New plan not found during upgrade', ['new_plan_id' => $request->New_Plan_id]);
            return response()->json(['status' => 'error', 'message' => 'New plan not found'], 404);
        }

        // Validate company relationship
        if (!$subscription->company) {
            Log::error('Company not found for subscription during upgrade', ['subscription_id' => $subscription->Subscription_id]);
            return response()->json(['status' => 'error', 'message' => 'Company not found for subscription'], 404);
        }

        // Log the payment request for debugging
        Log::info('Upgrade Payment Request', [
            'subscription_id' => $request->Subscription_id,
            'new_plan_id' => $request->New_Plan_id,
            'amount' => $request->Amount,
            'payment_gateway' => $request->Payment_gateway,
            'company_id' => $subscription->Company_id,
            'user_id' => $subscription->company->User_id,
        ]);

        // Prepare payment request
        $paymentRequest = new Request([
            'User_id' => $subscription->company->User_id,
            'Subscription_id' => $subscription->Subscription_id,
            'Amount' => $request->Amount,
            'Currency' => 'INR',
            'Payment_gateway' => $request->Payment_gateway,
        ]);

        $paymentController = new PaymentController();
        $paymentResponse = $paymentController->store($paymentRequest);

        // Log payment response for debugging
        Log::info('Payment Response', [
            'status_code' => $paymentResponse->getStatusCode(),
            'response_data' => $paymentResponse->getData(true),
        ]);

        // Check payment response more flexibly
        $responseData = $paymentResponse->getData(true);
        $isPaymentSuccessful = $paymentResponse->getStatusCode() === 200 &&
                             (isset($responseData['status']) && $responseData['status'] === 'success');

        if ($isPaymentSuccessful) {
            $paymentData = $responseData['data'] ?? null;
            if ($request->Payment_gateway === 'razorpay') {
                return response()->json([
                    'status' => 'pending',
                    'message' => 'Razorpay payment initiated',
                    'data' => $paymentData,
                ], 200);
            }

            // Use a transaction to ensure data consistency
            DB::beginTransaction();
            try {
                // Log the current subscription state to history before updating (OLD PLAN)
                $oldHistoryData = [
                    'subscription_id' => $subscription->Subscription_id,
                    'plan_id' => $subscription->Plan_id,
                    'company_id' => $subscription->Company_id,
                    'name' => $subscription->Name ?? 'Unnamed Subscription',
                    'start_date' => $subscription->Start_date->toDateTimeString(),
                    'end_date' => Carbon::now()->toDateTimeString(),
                    'status' => 'expired',
                    'amount' => $request->Amount,
                    'payment_gateway' => $request->Payment_gateway,
                ];
                $oldHistory = SubscriptionHistory::create($oldHistoryData);
                Log::info('Old subscription history logged', [
                    'history_id' => $oldHistory->history_id,
                    'data' => $oldHistoryData,
                ]);

                // Update the existing subscription with the new plan
                $subscription->update([
                    'Plan_id' => $request->New_Plan_id,
                    'Start_date' => Carbon::now(),
                    'End_date' => Carbon::now()->addDays($newPlan->Duration ?? 365),
                    'Status' => 'active',
                ]);

                // Log the new subscription state to history (NEW PLAN)
                $newHistoryData = [
                    'subscription_id' => $subscription->Subscription_id,
                    'plan_id' => $request->New_Plan_id,
                    'company_id' => $subscription->Company_id,
                    'name' => $subscription->Name ?? 'Unnamed Subscription',
                    'start_date' => $subscription->Start_date->toDateTimeString(),
                    'end_date' => $subscription->End_date->toDateTimeString(),
                    'status' => $subscription->Status,
                    'amount' => $request->Amount,
                    'payment_gateway' => $request->Payment_gateway,
                ];
                $newHistory = SubscriptionHistory::create($newHistoryData);
                Log::info('New subscription history logged', [
                    'history_id' => $newHistory->history_id,
                    'data' => $newHistoryData,
                ]);

                $companyId = $subscription->company->Company_id;
                $paymentController->syncUsageLimitsAfterPayment($companyId, $request->New_Plan_id);

                DB::commit();
                return response()->json([
                    'status' => 'success',
                    'message' => 'Subscription upgraded successfully',
                    'data' => $subscription->load('plan'),
                ], 200);
            } catch (\Exception $e) {
                DB::rollBack();
                Log::error('Failed to process subscription upgrade', [
                    'error' => $e->getMessage(),
                    'subscription_id' => $subscription->Subscription_id,
                    'old_history_data' => $oldHistoryData ?? null,
                    'new_history_data' => $newHistoryData ?? null,
                ]);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to process subscription upgrade',
                    'error' => $e->getMessage(),
                ], 500);
            }
        }

        // Log failed payment response
        Log::warning('Payment failed or not successful', [
            'status_code' => $paymentResponse->getStatusCode(),
            'response_data' => $paymentResponse->getData(true),
        ]);
        return $paymentResponse;
    }

    public function showUpgradeForm()
    {
        $plans = Plan::with('currency')->get();
        return view('upgrade-plan', compact('plans'));
    }

    public function getSubscriptionByName(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $subscriptions = Subscription::where('Name', $request->name)->get();

        if ($subscriptions->isEmpty()) {
            return response()->json(['message' => 'No subscriptions found with this name'], 404);
        }

        return response()->json($subscriptions);
    }

    public function getSubscriptionHistory($company_id): JsonResponse
    {
        $validator = Validator::make(['company_id' => $company_id], [
            'company_id' => 'required|integer|exists:company_profiles,Company_id',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $history = SubscriptionHistory::where('company_id', $company_id)
            ->with('plan')
            ->orderBy('start_date', 'desc')
            ->get();

        if ($history->isEmpty()) {
            return response()->json(['message' => 'No subscription history found'], 404);
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Subscription history retrieved successfully',
            'data' => $history
        ], 200);
    }
}
