<?php

namespace App\Http\Controllers;

use App\Models\Role;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class RoleController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:manage_roles')->except(['index', 'show']);
    }

    public function index(): JsonResponse
    {
        $roles = Role::all();
        return response()->json([
            'status' => 'success',
            'data' => $roles,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $role = Role::findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $role,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        Log::info('Store Role Request Data:', $request->all()); // Debug request payload

        $validator = Validator::make($request->all(), [
            'Name' => 'required|string|max:100|unique:Roles,Name',
            'Permissions' => 'required|array',
        ]);

        if ($validator->fails()) {
            Log::error('Validation Errors:', $validator->errors()->toArray()); // Log validation errors
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $role = Role::create([
            'Name' => $request->Name, // Use capitalized to match request and DB
            'Permissions' => $request->Permissions, // Use capitalized to match request and DB
        ]);

        return response()->json([
            'status' => 'success',
            'data' => $role,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $role = Role::findOrFail($id);

        Log::info('Update Role Request Data:', $request->all()); // Debug request payload

        $validator = Validator::make($request->all(), [
            'Name' => 'sometimes|string|max:100|unique:Roles,Name,' . $id . ',Role_id',
            'Permissions' => 'sometimes|array',
        ]);

        if ($validator->fails()) {
            Log::error('Validation Errors:', $validator->errors()->toArray()); // Log validation errors
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Correctly map and update only provided fields, fallback to existing values
        $updateData = [];
        if ($request->has('Name')) {
            $updateData['Name'] = $request->Name;
        }
        if ($request->has('Permissions')) {
            $updateData['Permissions'] = $request->Permissions;
        }
        $role->update($updateData);

        return response()->json([
            'status' => 'success',
            'data' => $role,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $role = Role::findOrFail($id);
        $role->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Role deleted successfully',
        ], 200);
    }
}
