<?php

namespace App\Http\Controllers;

use App\Models\Review;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class ReviewController extends Controller
{
    public function index(): JsonResponse
    {
        $reviews = Review::with(['user', 'job'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $reviews,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $review = Review::with(['user', 'job'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $review,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:Users,User_id',
            'Job_id' => 'required|exists:Jobs,Job_id',
            'Rating' => 'required|integer|between:1,5',
            'Comment' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $review = Review::create($request->only([
            'User_id',
            'Job_id',
            'Rating',
            'Comment',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $review,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $review = Review::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'User_id' => 'sometimes|exists:Users,User_id',
            'Job_id' => 'sometimes|exists:Jobs,Job_id',
            'Rating' => 'sometimes|integer|between:1,5',
            'Comment' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $review->update($request->only([
            'User_id',
            'Job_id',
            'Rating',
            'Comment',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $review,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $review = Review::findOrFail($id);
        $review->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Review deleted successfully',
        ], 200);
    }
}
