<?php

  namespace App\Http\Controllers;

  use App\Models\Report;
  use App\Models\Payment;
  use App\Models\Transaction;
  use App\Models\Refund;
  use App\Models\Analytics;
  use Illuminate\Http\Request;
  use Illuminate\Http\JsonResponse;
  use Illuminate\Support\Facades\Validator;

  class ReportsController extends Controller
  {
      public function index(): JsonResponse
      {
          $reports = Report::with('user')->get();
          return response()->json([
              'status' => 'success',
              'data' => $reports,
          ], 200);
      }

      public function show($id): JsonResponse
      {
          $report = Report::with('user')->findOrFail($id);
          return response()->json([
              'status' => 'success',
              'data' => $report,
          ], 200);
      }

      public function store(Request $request): JsonResponse
      {
          $validator = Validator::make($request->all(), [
              'User_id' => 'nullable|exists:Users,User_id',
              'Report_type' => 'required|in:payment_summary,user_activity',
              'Payment_status' => 'nullable|in:pending,completed,failed',
          ]);

          if ($validator->fails()) {
              return response()->json([
                  'status' => 'error',
                  'errors' => $validator->errors(),
              ], 422);
          }

          $data = $request->only(['User_id', 'Report_type', 'Payment_status']);
          $payments = Payment::query();

          if ($data['User_id']) {
              $payments->where('User_id', $data['User_id']);
          }
          if (isset($data['Payment_status'])) {
              $payments->where('Payment_status', $data['Payment_status']);
          }

          $totalAmount = $payments->sum('Amount');
          $transactionCount = $payments->count();

          $report = Report::create([
              'User_id' => $data['User_id'],
              'Report_type' => $data['Report_type'],
              'Total_amount' => $totalAmount,
              'Transaction_count' => $transactionCount,
              'Payment_status' => $data['Payment_status'],
          ]);

          return response()->json([
              'status' => 'success',
              'data' => $report,
          ], 201);
      }

      public function generateReport(Request $request): JsonResponse
      {
          $validator = Validator::make($request->all(), [
              'User_id' => 'nullable|exists:Users,User_id',
              'Payment_status' => 'nullable|in:pending,completed,failed',
              'start_date' => 'nullable|date',
              'end_date' => 'nullable|date|after_or_equal:start_date',
          ]);

          if ($validator->fails()) {
              return response()->json([
                  'status' => 'error',
                  'errors' => $validator->errors(),
              ], 422);
          }

          $query = Payment::query()
              ->with(['transactions', 'refunds'])
              ->selectRaw('SUM(Payments.Amount) as total_payments, COUNT(*) as payment_count')
              ->leftJoin('Transactions', 'Payments.Payment_id', '=', 'Transactions.Payment_id')
              ->leftJoin('Refunds', 'Payments.Payment_id', '=', 'Refunds.Payment_id');

          if ($request->User_id) {
              $query->where('Payments.User_id', $request->User_id);
          }
          if ($request->Payment_status) {
              $query->where('Payments.Payment_status', $request->Payment_status);
          }
          if ($request->start_date && $request->end_date) {
              $query->whereBetween('Payments.Created_at', [$request->start_date, $request->end_date]);
          } elseif ($request->start_date) {
              $query->where('Payments.Created_at', '>=', $request->start_date);
          }

          $report = $query->groupBy('Payments.Payment_id')
              ->get()
              ->reduce(function ($carry, $item) {
                  $carry['total_payments'] = ($carry['total_payments'] ?? 0) + $item->total_payments;
                  $carry['payment_count'] = ($carry['payment_count'] ?? 0) + $item->payment_count;
                  $carry['total_refunded'] = ($carry['total_refunded'] ?? 0) + optional($item->refunds)->sum('Amount');
                  $carry['transaction_count'] = ($carry['transaction_count'] ?? 0) + $item->transactions->count();
                  return $carry;
              }, []);

          return response()->json([
              'status' => 'success',
              'data' => $report,
          ], 200);
      }

      public function update(Request $request, $id): JsonResponse
      {
          $report = Report::findOrFail($id);

          $validator = Validator::make($request->all(), [
              'User_id' => 'sometimes|exists:Users,User_id',
              'Report_type' => 'sometimes|in:payment_summary,user_activity',
              'Total_amount' => 'sometimes|numeric|min:0',
              'Transaction_count' => 'sometimes|integer|min:0',
              'Payment_status' => 'sometimes|in:pending,completed,failed',
          ]);

          if ($validator->fails()) {
              return response()->json([
                  'status' => 'error',
                  'errors' => $validator->errors(),
              ], 422);
          }

          $report->update($request->only([
              'User_id',
              'Report_type',
              'Total_amount',
              'Transaction_count',
              'Payment_status',
          ]));

          return response()->json([
              'status' => 'success',
              'data' => $report,
          ], 200);
      }

      public function destroy($id): JsonResponse
      {
          $report = Report::findOrFail($id);
          $report->delete();

          return response()->json([
              'status' => 'success',
              'message' => 'Report deleted successfully',
          ], 200);
      }

      public function storeAnalytics(Request $request): JsonResponse
      {
          $validator = Validator::make($request->all(), [
              'Type' => 'required|in:user_engagement,job_placement,training_stats',
              'Data' => 'required|json',
          ]);

          if ($validator->fails()) {
              return response()->json([
                  'status' => 'error',
                  'errors' => $validator->errors(),
              ], 422);
          }

          $analytics = Analytics::create([
              'Type' => $request->Type,
              'Data' => $request->Data,
          ]);

          return response()->json([
              'status' => 'success',
              'data' => $analytics,
          ], 201);
      }
  }
