<?php

namespace App\Http\Controllers;

use App\Models\PracticeTest;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class PracticeTestController extends Controller
{
    public function index(): JsonResponse
    {
        $practiceTests = PracticeTest::with(['lesson', 'user'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $practiceTests,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Lesson_id' => 'required|exists:lessons,Lesson_id',
            'User_id' => 'required|exists:users,User_id',
            'Score' => 'required|numeric|between:0,100',
            'Attempt_date' => 'required|date',
            'Passed' => 'sometimes|boolean',
            'Total_questions' => 'required|integer|min:0',
            'Difficulty_level' => 'required|in:easy,medium,hard',
            'Duration_minutes' => 'nullable|integer|min:1',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $practiceTest = PracticeTest::create($request->only([
            'Lesson_id',
            'User_id',
            'Score',
            'Attempt_date',
            'Passed',
            'Total_questions',
            'Difficulty_level',
            'Duration_minutes',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $practiceTest,
        ], 201);
    }

    public function show($id): JsonResponse
    {
        $practiceTest = PracticeTest::with(['lesson', 'user'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $practiceTest,
        ], 200);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $practiceTest = PracticeTest::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Lesson_id' => 'sometimes|exists:lessons,Lesson_id',
            'User_id' => 'sometimes|exists:users,User_id',
            'Score' => 'sometimes|numeric|between:0,100',
            'Attempt_date' => 'sometimes|date',
            'Passed' => 'sometimes|boolean',
            'Total_questions' => 'sometimes|integer|min:0',
            'Difficulty_level' => 'sometimes|in:easy,medium,hard',
            'Duration_minutes' => 'sometimes|nullable|integer|min:1',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $practiceTest->update($request->only([
            'Lesson_id',
            'User_id',
            'Score',
            'Attempt_date',
            'Passed',
            'Total_questions',
            'Difficulty_level',
            'Duration_minutes',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $practiceTest,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $practiceTest = PracticeTest::findOrFail($id);
        $practiceTest->delete();
        return response()->json([
            'status' => 'success',
            'message' => 'Practice test deleted successfully',
        ], 204);
    }
}
?>
