<?php

namespace App\Http\Controllers;

use App\Models\Plan;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class PlanController extends Controller
{
    public function index(): JsonResponse
    {
        $plans = Plan::with('currency')->get();
        return response()->json([
            'status' => 'success',
            'data' => $plans,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $plan = Plan::with('currency')->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $plan,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Name' => 'required|string|max:255|unique:plans,Name',
            'Features' => 'required|array',
            'Features.features' => 'required|array',
            'Features.features.*' => 'string',
            'Price' => 'required|numeric|min:0',
            'Currency_id' => 'required|exists:currencies,Currency_id',
            'Duration' => 'required|integer|min:0',
            'Status' => 'required|in:active,inactive',
            'Access_Level' => 'required|in:basic,premium,enterprise',
            'Max_Job_Postings' => 'required|integer|min:0',
            'Max_Interview_Schedules' => 'required|integer|min:0',
        ], [
            'Name.unique' => 'The plan name is already taken.',
            'Currency_id.exists' => 'The specified currency does not exist.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $plan = Plan::create([
            'Name' => $request->Name,
            'Features' => $request->Features,
            'Price' => $request->Price,
            'Currency_id' => $request->Currency_id,
            'Duration' => $request->Duration,
            'Status' => $request->Status,
            'Access_Level' => $request->Access_Level,
            'Max_Job_Postings' => $request->Max_Job_Postings,
            'Max_Interview_Schedules' => $request->Max_Interview_Schedules,
        ]);

        return response()->json([
            'status' => 'success',
            'data' => $plan->load('currency'),
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $plan = Plan::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Name' => 'sometimes|string|max:255',
            'Features' => 'sometimes|array',
            'Features.features' => 'sometimes|array',
            'Features.features.*' => 'sometimes|string',
            'Price' => 'sometimes|numeric|min:0',
            'Currency_id' => 'sometimes|exists:currencies,Currency_id',
            'Duration' => 'sometimes|integer|min:0',
            'Status' => 'sometimes|in:active,inactive',
            'Access_Level' => 'sometimes|in:basic,premium,enterprise',
            'Max_Job_Postings' => 'sometimes|integer|min:0',
            'Max_Interview_Schedules' => 'sometimes|integer|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Validate JSON structure for Features if provided
        if ($request->has('Features')) {
            $features = json_decode(json_encode($request->Features), true); // Convert array to JSON and decode
            if (!is_array($features)) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Features must be a valid array',
                ], 422);
            }
        }

        $plan->update($request->only([
            'Name',
            'Features',
            'Price',
            'Currency_id',
            'Duration',
            'Status',
            'Access_Level',
            'Max_Job_Postings',
            'Max_Interview_Schedules',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $plan->load('currency'),
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $plan = Plan::findOrFail($id);
        $plan->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Plan deleted successfully',
        ], 200);
    }
}
