<?php

namespace App\Http\Controllers;

use App\Models\Course;
use App\Models\Payment;
use App\Models\UserCourse;
use App\Models\Transaction;
use App\Models\Coupon;
use App\Models\CompanyProfile;
use App\Models\User;
use App\Services\PaymentHandler;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class PaymentController extends Controller
{
    public function index(): JsonResponse
    {
        $payments = Payment::with(['user', 'subscription', 'coupon'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $payments,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $payment = Payment::with(['user', 'subscription', 'coupon'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $payment,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:users,User_id',
            'Subscription_id' => 'nullable|exists:subscriptions,Subscription_id',
            'Coupon_id' => 'nullable|exists:coupons,Coupon_id',
            'Course_id' => 'required_without:Subscription_id|exists:courses,Course_id',
            'Amount' => 'nullable|numeric|min:0',
            'Currency' => 'nullable|string|size:3',
            'Payment_gateway' => 'required_if:Is_paid,1|in:paypal,stripe,razorpay',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => 'error', 'errors' => $validator->errors()], 422);
        }

        // Check if course exists and is free
        if ($request->Course_id) {
            $course = Course::find($request->Course_id);
            if (!$course) {
                return response()->json(['status' => 'error', 'message' => 'Course not found'], 404);
            }

            if ($course->Is_paid == 0) {
                return DB::transaction(function () use ($request, $course) {
                    $payment = Payment::create([
                        'User_id' => $request->User_id,
                        'Course_id' => $request->Course_id,
                        'Amount' => 0.00,
                        'Currency' => $request->Currency ?? 'USD',
                        'Payment_status' => 'completed',
                        'Payment_gateway' => 'free',
                    ]);

                    $existingEnrollment = UserCourse::where('User_id', $request->User_id)
                        ->where('Course_id', $request->Course_id)
                        ->first();
                    if (!$existingEnrollment) {
                        UserCourse::create([
                            'User_id' => $request->User_id,
                            'Course_id' => $request->Course_id,
                            'Enrollment_number' => Str::uuid()->toString(),
                            'Status' => 'enrolled',
                            'Payment_id' => $payment->Payment_id,
                            'Created_at' => now(),
                            'Updated_at' => now(),
                        ]);
                    }

                    return response()->json([
                        'status' => 'success',
                        'message' => 'Free course enrolled successfully',
                        'data' => $payment->fresh(),
                    ], 201);
                });
            }
        }

        $discountedAmount = $request->Amount ?? 0;
        $coupon = null;
        if ($request->Coupon_id) {
            $coupon = Coupon::where('Coupon_id', $request->Coupon_id)
                ->where('Status', 'active')
                ->where('Expiry_date', '>=', now()->toDateString())
                ->where(function ($query) {
                    $query->where('Usage_limit', 0)
                        ->orWhereColumn('Used_count', '<', 'Usage_limit');
                })
                ->first();

            if (!$coupon) {
                return response()->json(['status' => 'error', 'message' => 'Invalid or expired coupon'], 422);
            }

            if ($coupon->Plan_id && $request->Subscription_id) {
                $subscription = \App\Models\Subscription::find($request->Subscription_id);
                if (!$subscription) {
                    return response()->json(['status' => 'error', 'message' => 'Invalid subscription'], 422);
                }
                if ($coupon->Plan_id !== $subscription->Plan_id) {
                    return response()->json(['status' => 'error', 'message' => 'Coupon not applicable for this subscription'], 422);
                }
            }

            if ($coupon->Discount_type === 'percentage') {
                $discount = ($coupon->Discount_value / 100) * $discountedAmount;
                $discountedAmount -= $discount;
            } else {
                $discountedAmount -= $coupon->Discount_value;
            }

            if ($discountedAmount < 0) {
                $discountedAmount = 0;
            }
        }

        $gatewayAmount = round($discountedAmount * 100);

        return DB::transaction(function () use ($request, $discountedAmount, $gatewayAmount, $coupon) {
            $payment = Payment::create([
                'User_id' => $request->User_id,
                'Subscription_id' => $request->Subscription_id,
                'Coupon_id' => $request->Coupon_id,
                'Course_id' => $request->Course_id,
                'Amount' => $discountedAmount,
                'Currency' => $request->Currency ?? 'USD',
                'Payment_gateway' => $request->Payment_gateway,
                'Payment_status' => 'pending',
            ]);

            try {
                $paymentHandler = new PaymentHandler($request->Payment_gateway);
                $response = $paymentHandler->initiatePayment($payment, ['amount' => $gatewayAmount]);

                $payment->update(['Transaction_id' => $response['transaction_id'] ?? null]);

                if ($coupon) {
                    $coupon->increment('Used_count');
                }

                return response()->json([
                    'status' => 'success',
                    'data' => $payment->fresh(),
                    'gateway_response' => $response,
                    'RAZORPAY_KEY' => config('services.razorpay.key'),
                ], 201);
            } catch (\Exception $e) {
                Log::error('Payment initiation failed', ['error' => $e->getMessage(), 'payment_id' => $payment->Payment_id]);
                $payment->update(['Payment_status' => 'failed', 'Transaction_id' => null]);
                return response()->json(['status' => 'error', 'message' => 'Payment initiation failed: ' . $e->getMessage()], 500);
            }
        });
    }

    public function update(Request $request, $id): JsonResponse
    {
        $userId = Auth::id();
        $payment = Payment::where('User_id', $userId)->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'User_id' => 'sometimes|exists:users,User_id',
            'Subscription_id' => 'sometimes|exists:subscriptions,Subscription_id',
            'Coupon_id' => 'sometimes|nullable|exists:coupons,Coupon_id',
            'Amount' => 'sometimes|numeric|min:0',
            'Currency' => 'sometimes|string|size:3',
            'Payment_status' => 'sometimes|in:pending,completed,failed',
            'Payment_gateway' => 'sometimes|in:paypal,stripe,razorpay',
            'Transaction_id' => 'sometimes|string|max:255',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        if ($request->has('Coupon_id') && $request->Coupon_id !== $payment->Coupon_id) {
            $coupon = Coupon::where('Coupon_id', $request->Coupon_id)
                ->where('Status', 'active')
                ->where('Expiry_date', '>=', now()->toDateString())
                ->where(function ($query) {
                    $query->where('Usage_limit', 0)
                        ->orWhereColumn('Used_count', '<', 'Usage_limit');
                })
                ->first();

            if (!$coupon) {
                return response()->json(['status' => 'error', 'message' => 'Invalid or expired coupon'], 422);
            }

            if ($coupon->Plan_id && $request->Subscription_id) {
                $subscription = \App\Models\Subscription::find($request->Subscription_id);
                if (!$subscription) {
                    return response()->json(['status' => 'error', 'message' => 'Invalid subscription'], 422);
                }
                if ($coupon->Plan_id !== $subscription->Plan_id) {
                    return response()->json(['status' => 'error', 'message' => 'Coupon not applicable for this subscription'], 422);
                }
            }

            if ($request->has('Amount')) {
                $discountedAmount = $request->Amount;
                if ($coupon->Discount_type === 'percentage') {
                    $discount = ($coupon->Discount_value / 100) * $request->Amount;
                    $discountedAmount -= $discount;
                } else {
                    $discountedAmount -= $coupon->Discount_value;
                }
                $request->merge(['Amount' => max(0, $discountedAmount)]);
            }
        }

        $payment->update($request->only([
            'User_id',
            'Subscription_id',
            'Coupon_id',
            'Amount',
            'Currency',
            'Payment_status',
            'Payment_gateway',
            'Transaction_id',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $payment->fresh(),
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $userId = Auth::id();
        $payment = Payment::where('User_id', $userId)->findOrFail($id);
        $payment->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Payment deleted successfully',
        ], 200);
    }

  public function callback(Request $request, $gateway): JsonResponse
{
    $validator = Validator::make($request->all(), ['Transaction_id' => 'required|string']);
    if ($validator->fails()) {
        return response()->json(['status' => 'error', 'errors' => $validator->errors()], 422);
    }

    return DB::transaction(function () use ($request, $gateway) {
        try {
            Log::debug('Querying payments table', [
                'table' => (new Payment)->getTable(),
                'transaction_id' => $request->Transaction_id,
            ]);

            $paymentHandler = new PaymentHandler($gateway);
            $gatewayData = $request->only(['razorpay_payment_id', 'razorpay_signature']);
            $verified = $paymentHandler->verifyPayment($request->Transaction_id, $gatewayData);

            if (!$verified) {
                Log::warning('Payment verification failed', [
                    'transaction_id' => $request->Transaction_id,
                    'gateway' => $gateway,
                ]);
                return response()->json(['status' => 'error', 'message' => 'Payment verification failed'], 400);
            }

            $payment = Payment::where('Transaction_id', $request->Transaction_id)
                ->lockForUpdate()
                ->firstOrFail();

            $payment->update(['Payment_status' => 'completed']);

            $transaction = Transaction::where('Payment_id', $payment->Payment_id)->first();

            if ($transaction) {
                $transaction->update([
                    'Transaction_type' => 'Course Payment',
                    'Status' => 'completed',
                    'Gateway_response' => json_encode($gatewayData),
                    'Processed_at' => now(),
                ]);
            } else {
                $transaction = Transaction::create([
                    'Payment_id' => $payment->Payment_id,
                    'Transaction_type' => 'Course Payment',
                    'Amount' => $payment->Amount,
                    'Currency' => $payment->Currency,
                    'Payment_method' => $payment->Payment_gateway,
                    'Status' => 'completed',
                    'Gateway_response' => json_encode($gatewayData),
                    'Processed_at' => now(),
                ]);
            }

            if ($payment->Course_id) {
                $existingEnrollment = UserCourse::where('User_id', $payment->User_id)
                    ->where('Course_id', $payment->Course_id)
                    ->first();
                if (!$existingEnrollment) {
                    Log::debug('Creating new UserCourse enrollment', [
                        'User_id' => $payment->User_id,
                        'Course_id' => $payment->Course_id,
                        'Payment_id' => $payment->Payment_id,
                    ]);
                    UserCourse::create([
                        'User_id' => $payment->User_id,
                        'Course_id' => $payment->Course_id,
                        'Enrollment_number' => Str::uuid()->toString(),
                        'Status' => 'enrolled',
                        'Payment_id' => $payment->Payment_id,
                        'Created_at' => now(),
                        'Updated_at' => now(),
                    ]);
                }
            }

            return response()->json([
                'status' => 'success',
                'message' => 'Payment verified successfully',
                'data' => $payment->fresh(),
                'transaction' => $transaction,
            ], 200);
        } catch (\Illuminate\Database\QueryException $e) {
            Log::error('Database error in payment callback', [
                'error' => $e->getMessage(),
                'transaction_id' => $request->Transaction_id,
                'gateway' => $gateway,
                'sql' => $e->getSql(),
                'bindings' => $e->getBindings(),
            ]);
            return response()->json(['status' => 'error', 'message' => 'Database error: ' . $e->getMessage()], 500);
        } catch (\Exception $e) {
            Log::error('Unexpected error in callback', [
                'error' => $e->getMessage(),
                'transaction_id' => $request->Transaction_id,
                'gateway' => $gateway,
            ]);
            return response()->json(['status' => 'error', 'message' => 'Unexpected error: ' . $e->getMessage()], 500);
        }
    }, 5);
}

    public function totalRevenue(): JsonResponse
    {
        $courseRevenue = Payment::whereNotNull('Course_id')
            ->where('Payment_status', 'completed')
            ->sum('Amount');

        $planRevenue = Payment::whereNotNull('Subscription_id')
            ->where('Payment_status', 'completed')
            ->sum('Amount');

        $totalRevenue = $courseRevenue + $planRevenue;

        $totalCandidates = User::where('User_type', 'candidate')->count();
        $totalCompanies = CompanyProfile::count();

        return response()->json([
            'status' => 'success',
            'data' => [
                'course_revenue' => $courseRevenue,
                'plan_revenue' => $planRevenue,
                'total_revenue' => $totalRevenue,
                'total_candidates' => $totalCandidates,
                'total_companies' => $totalCompanies,
            ],
        ], 200);
    }
}