<?php

namespace App\Http\Controllers;

use App\Models\Message;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class MessageController extends Controller
{
    public function index(): JsonResponse
    {
        $messages = Message::with(['sender', 'receiver'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $messages,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $message = Message::with(['sender', 'receiver'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $message,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Sender_id' => 'required|exists:Users,User_id',
            'Receiver_id' => 'required|exists:Users,User_id',
            'Content' => 'required|string',
            'Is_read' => 'sometimes|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $message = Message::create($request->only([
            'Sender_id',
            'Receiver_id',
            'Content',
            'Is_read',
        ]));

        broadcast(new \App\Events\NewMessage($message));

        return response()->json([
            'status' => 'success',
            'data' => $message,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $message = Message::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Sender_id' => 'sometimes|exists:Users,User_id',
            'Receiver_id' => 'sometimes|exists:Users,User_id',
            'Content' => 'sometimes|string',
            'Is_read' => 'sometimes|boolean',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $message->update($request->only([
            'Sender_id',
            'Receiver_id',
            'Content',
            'Is_read',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $message,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $message = Message::findOrFail($id);
        $message->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Message deleted successfully',
        ], 200);
    }
}
