<?php

namespace App\Http\Controllers;

use App\Models\Lesson;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class LessonController extends Controller
{
    public function index(): JsonResponse
    {
        $lessons = Lesson::with('section')->get();
        return response()->json([
            'status' => 'success',
            'data' => $lessons,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $lesson = Lesson::with('section')->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $lesson,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Section_id' => 'required|exists:sections,Section_id',
            'Type' => 'required|in:video,quiz,assignment,document',
            'Title' => 'required|string|max:255',
            'Content_url' => 'sometimes|file|max:1048576', // Allow up to 1GB before compression
            'Video_link' => 'sometimes|nullable|url|regex:/^https:\/\/www\.youtube\.com\/watch\?v=[\w-]+$/',
            'Duration' => 'sometimes|integer|nullable',
            'Order' => 'sometimes|integer',
            'Metadata' => 'sometimes|json',
            'Has_assignments' => 'sometimes|boolean',
            'Has_practice_tests' => 'sometimes|boolean',
            'Has_mock_interviews' => 'sometimes|boolean',
            'Has_interactive_elements' => 'sometimes|boolean',
        ]);

        if ($validator->fails()) {
            Log::error('Validation failed for store request', ['errors' => $validator->errors(), 'request' => $request->all()]);
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'Section_id',
            'Type',
            'Title',
            'Content_url',
            'Video_link',
            'Duration',
            'Order',
            'Metadata',
            'Has_assignments',
            'Has_practice_tests',
            'Has_mock_interviews',
            'Has_interactive_elements',
        ]);

        if ($request->hasFile('Content_url')) {
            $content = $request->file('Content_url');
            $validMimes = [
                'video/mp4',
                'video/x-msvideo',
                'video/quicktime',
                'application/pdf',
                'application/msword',
                'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            ];

            Log::info("Processing file: {$content->getClientOriginalName()}");
            if ($content->isValid()) {
                $fileMime = $content->getMimeType();
                if (!in_array($fileMime, $validMimes)) {
                    return response()->json([
                        'status' => 'error',
                        'message' => "Invalid MIME type: $fileMime, expected: " . implode(', ', $validMimes),
                    ], 422);
                }

                $fileSize = $content->getSize();
                $isVideo = in_array($fileMime, ['video/mp4', 'video/x-msvideo', 'video/quicktime']);
                $contentData = file_get_contents($content->getRealPath());

                if ($isVideo && $fileSize > 102400 * 1024) {
                    try {
                        $contentData = $this->compressVideo($contentData, $fileMime, $content->getClientOriginalName());
                        if (strlen($contentData) > 102400 * 1024) {
                            return response()->json([
                                'status' => 'error',
                                'message' => "Compressed video {$content->getClientOriginalName()} still exceeds 100MB",
                            ], 422);
                        }
                    } catch (\Exception $e) {
                        Log::error("Compression failed for file: {$content->getClientOriginalName()}", ['error' => $e->getMessage()]);
                        return response()->json([
                            'status' => 'error',
                            'message' => "Compression failed: " . $e->getMessage(),
                        ], 422);
                    }
                }

                $contentName = time() . '_' . str_replace(' ', '_', pathinfo($content->getClientOriginalName(), PATHINFO_FILENAME)) . '.' . $content->getClientOriginalExtension();
                $contentPath = $content->storeAs('lesson_contents', $contentName, 'public');
                if ($contentPath) {
                    $data['Content_url'] = url('storage/lesson_contents/' . $contentName);
                    Log::info("Successfully stored: {$content->getClientOriginalName()} at {$contentPath}");

                    // Compute duration for uploaded videos
                    if ($isVideo) {
                        $inputPath = storage_path('app/public/lesson_contents/' . $contentName);
                        $command = "ffprobe -v error -show_entries format=duration -of default=noprint_wrappers=1:nokey=1 {$inputPath}";
                        $durationSeconds = (float) shell_exec($command);
                        $data['Duration'] = round($durationSeconds / 60); // Set in minutes
                    }
                } else {
                    Log::error("Failed to store file: {$content->getClientOriginalName()}");
                    return response()->json([
                        'status' => 'error',
                        'message' => "Failed to store file: {$content->getClientOriginalName()}",
                    ], 422);
                }
            } else {
                Log::warning("Invalid file uploaded: {$content->getClientOriginalName()}");
                return response()->json([
                    'status' => 'error',
                    'message' => "Invalid file: {$content->getClientOriginalName()}",
                ], 422);
            }
        }

        // Set duration for YouTube videos
        if (isset($data['Video_link']) && !isset($data['Duration'])) {
            $data['Duration'] = $this->getYouTubeDuration($data['Video_link']); // Returns minutes
        }

        $lesson = Lesson::create($data);

        return response()->json([
            'status' => 'success',
            'data' => $lesson,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
{
    $lesson = Lesson::findOrFail($id);

    $validator = Validator::make($request->all(), [
        'Section_id' => 'sometimes|exists:sections,Section_id',
        'Type' => 'sometimes|in:video,quiz,assignment,document',
        'Title' => 'sometimes|string|max:255',
        'Content_url' => 'sometimes|file|max:1048576', // Allow up to 1GB
        'Video_link' => ['sometimes', 'nullable', function ($attribute, $value, $fail) {
            if ($value && !preg_match('/^data:[\w\/]+;base64,/', $value) && !preg_match('/^https:\/\/www\.youtube\.com\/watch\?v=[\w-]+$/', $value)) {
                $fail('The ' . $attribute . ' must be a valid YouTube URL or a base64 string.');
            }
        }],
        'Duration' => 'sometimes|integer|nullable',
        'Order' => 'sometimes|integer',
        'Metadata' => 'sometimes|json',
        'Has_assignments' => 'sometimes|boolean',
        'Has_practice_tests' => 'sometimes|boolean',
        'Has_mock_interviews' => 'sometimes|boolean',
        'Has_interactive_elements' => 'sometimes|boolean',
    ]);

    if ($validator->fails()) {
        Log::error('Validation failed for update request', ['errors' => $validator->errors(), 'request' => $request->all()]);
        return response()->json([
            'status' => 'error',
            'errors' => $validator->errors(),
        ], 422);
    }

    $data = $request->only([
        'Section_id',
        'Type',
        'Title',
        'Content_url',
        'Video_link',
        'Duration',
        'Order',
        'Metadata',
        'Has_assignments',
        'Has_practice_tests',
        'Has_mock_interviews',
        'Has_interactive_elements',
    ]);

    if ($request->hasFile('Content_url')) {
        $content = $request->file('Content_url');
        $validMimes = [
            'video/mp4',
            'video/x-msvideo',
            'video/quicktime',
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        ];

        Log::info("Processing file: {$content->getClientOriginalName()}");
        if ($content->isValid()) {
            $fileMime = $content->getMimeType();
            if (!in_array($fileMime, $validMimes)) {
                return response()->json([
                    'status' => 'error',
                    'message' => "Invalid MIME type: $fileMime, expected: " . implode(', ', $validMimes),
                ], 422);
            }

            $fileSize = $content->getSize();
            $isVideo = in_array($fileMime, ['video/mp4', 'video/x-msvideo', 'video/quicktime']);
            $contentData = file_get_contents($content->getRealPath());

            if ($isVideo && $fileSize > 102400 * 1024) {
                try {
                    $contentData = $this->compressVideo($contentData, $fileMime, $content->getClientOriginalName());
                    if (strlen($contentData) > 102400 * 1024) {
                        return response()->json([
                            'status' => 'error',
                            'message' => "Compressed video {$content->getClientOriginalName()} still exceeds 100MB",
                        ], 422);
                    }
                } catch (\Exception $e) {
                    Log::error("Compression failed for file: {$content->getClientOriginalName()}", ['error' => $e->getMessage()]);
                    return response()->json([
                        'status' => 'error',
                        'message' => "Compression failed: " . $e->getMessage(),
                    ], 422);
                }
            }

            $contentName = time() . '_' . str_replace(' ', '_', pathinfo($content->getClientOriginalName(), PATHINFO_FILENAME)) . '.' . $content->getClientOriginalExtension();
            $contentPath = $content->storeAs('lesson_contents', $contentName, 'public');
            if ($contentPath) {
                if ($lesson->Content_url) {
                    $contentPathOld = str_replace(url('storage/'), 'public/', $lesson->Content_url);
                    if (Storage::exists($contentPathOld)) {
                        Storage::delete($contentPathOld);
                    }
                }
                $data['Content_url'] = url('storage/lesson_contents/' . $contentName);
                Log::info("Successfully stored: {$content->getClientOriginalName()} at {$contentPath}");

                // Compute duration for uploaded videos
                if ($isVideo) {
                    $inputPath = storage_path('app/public/lesson_contents/' . $contentName);
                    $command = "ffprobe -v error -show_entries format=duration -of default=noprint_wrappers=1:nokey=1 {$inputPath}";
                    $durationSeconds = (float) shell_exec($command);
                    $data['Duration'] = round($durationSeconds / 60); // Set in minutes
                }
            } else {
                Log::error("Failed to store file: {$content->getClientOriginalName()}");
                return response()->json([
                    'status' => 'error',
                    'message' => "Failed to store file: {$content->getClientOriginalName()}",
                ], 422);
            }
        } else {
            Log::warning("Invalid file uploaded: {$content->getClientOriginalName()}");
            return response()->json([
                'status' => 'error',
                'message' => "Invalid file: {$content->getClientOriginalName()}",
            ], 422);
        }
    } elseif ($request->input('Content_url') && $this->isValidBase64($request->input('Content_url'))) {
        $base64String = $request->input('Content_url');
        $fileMime = $this->getMimeTypeFromBase64($base64String);
        $validMimes = [
            'video/mp4',
            'video/x-msvideo',
            'video/quicktime',
            'application/pdf',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
        ];

        if (!in_array($fileMime, $validMimes)) {
            Log::error("Invalid base64 MIME type: {$fileMime}");
            return response()->json([
                'status' => 'error',
                'message' => "Invalid base64 MIME type: {$fileMime}, expected: " . implode(', ', $validMimes),
            ], 422);
        }

        $base64Data = preg_replace('/^data:[\w\/]+;base64,/', '', $base64String);
        $contentData = base64_decode($base64Data);
        if ($contentData === false) {
            Log::error("Failed to decode base64 string");
            return response()->json([
                'status' => 'error',
                'message' => 'Invalid base64 data',
            ], 422);
        }

        $isVideo = in_array($fileMime, ['video/mp4', 'video/x-msvideo', 'video/quicktime']);
        $extension = [
            'video/mp4' => 'mp4',
            'video/x-msvideo' => 'avi',
            'video/quicktime' => 'mov',
            'application/pdf' => 'pdf',
            'application/msword' => 'doc',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'docx',
        ][$fileMime] ?? 'bin';

        if ($isVideo && strlen($contentData) > 102400 * 1024) {
            try {
                $contentData = $this->compressVideo($contentData, $fileMime, "base64_upload.{$extension}");
                if (strlen($contentData) > 102400 * 1024) {
                    return response()->json([
                        'status' => 'error',
                        'message' => "Compressed base64 video still exceeds 100MB",
                    ], 422);
                }
            } catch (\Exception $e) {
                Log::error("Compression failed for base64 video", ['error' => $e->getMessage()]);
                return response()->json([
                    'status' => 'error',
                    'message' => "Compression failed: " . $e->getMessage(),
                ], 422);
            }
        }

        $contentName = time() . '_base64_upload.' . $extension;
        $contentPath = 'lesson_contents/' . $contentName;
        if (Storage::disk('public')->put($contentPath, $contentData)) {
            if ($lesson->Content_url) {
                $contentPathOld = str_replace(url('storage/'), 'public/', $lesson->Content_url);
                if (Storage::exists($contentPathOld)) {
                    Storage::delete($contentPathOld);
                }
            }
            $data['Content_url'] = url('storage/lesson_contents/' . $contentName);
            Log::info("Successfully stored base64 content at {$contentPath}");

            // Compute duration for uploaded videos
            if ($isVideo) {
                $inputPath = storage_path('app/public/lesson_contents/' . $contentName);
                $command = "ffprobe -v error -show_entries format=duration -of default=noprint_wrappers=1:nokey=1 {$inputPath}";
                $durationSeconds = (float) shell_exec($command);
                $data['Duration'] = round($durationSeconds / 60); // Set in minutes
            }
        } else {
            Log::error("Failed to store base64 content");
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to store base64 content',
            ], 422);
        }
    }

    // Set duration for YouTube videos
    if (isset($data['Video_link']) && !isset($data['Duration'])) {
        $data['Duration'] = $this->getYouTubeDuration($data['Video_link']); // Returns minutes
    }

    $lesson->update($data);

    return response()->json([
        'status' => 'success',
        'data' => $lesson,
    ], 200);
}

    public function destroy($id): JsonResponse
    {
        $lesson = Lesson::findOrFail($id);
        if ($lesson->Content_url) {
            $contentPath = str_replace(url('storage/'), 'public/', $lesson->Content_url);
            if (Storage::exists($contentPath)) {
                Storage::delete($contentPath);
            }
        }

        $lesson->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Lesson deleted successfully',
        ], 200);
    }

    public function deleteContentUrl($id): JsonResponse
    {
        $lesson = Lesson::findOrFail($id);
        if (!$lesson->Content_url) {
            return response()->json([
                'status' => 'error',
                'message' => 'No content URL to delete',
            ], 400);
        }

        $contentPath = str_replace(url('storage/'), 'public/', $lesson->Content_url);
        if (Storage::exists($contentPath)) {
            Storage::delete($contentPath);
        }

        $lesson->update(['Content_url' => null]);

        return response()->json([
            'status' => 'success',
            'message' => 'Content URL deleted successfully',
            'data' => $lesson,
        ], 200);
    }

    /**
     * Helper method to get MIME type from base64 string
     */
    private function getMimeTypeFromBase64($base64String)
    {
        if (preg_match('/^data:([\w\/]+);base64,/', $base64String, $matches)) {
            $mimeType = $matches[1];
            $validMimes = [
                'video/mp4',
                'video/x-msvideo',
                'video/quicktime',
                'application/pdf',
                'application/msword',
                'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
            ];
            return in_array($mimeType, $validMimes) ? $mimeType : 'invalid';
        }

        $base64Data = preg_replace('/^data:[\w\/]+;base64,/', '', $base64String);
        $decodedData = base64_decode($base64Data, true);

        if ($decodedData === false || strlen($decodedData) < 12) {
            Log::warning("Failed to decode base64 or insufficient data: " . substr($base64String, 0, 50) . "...");
            return 'invalid';
        }

        if (substr($decodedData, 0, 4) === '%PDF') {
            return 'application/pdf';
        }
        if (substr($decodedData, 4, 8) === 'ftypmp4') {
            return 'video/mp4';
        }
        if (substr($decodedData, 4, 6) === 'ftypqt' && substr($decodedData, 0, 4) === "\x00\x00\x00\x18") {
            return 'video/quicktime';
        }
        if (substr($decodedData, 0, 4) === 'RIFF' && substr($decodedData, 8, 4) === 'AVI ') {
            return 'video/x-msvideo';
        }
        if (substr($decodedData, 0, 2) === "\xD0\xCF" && substr($decodedData, 512, 4) === "\xEC\xA5\xC1\x00") {
            return 'application/msword';
        }
        if (substr($decodedData, 0, 4) === 'PK' && substr($decodedData, 30, 4) === 'word') {
            return 'application/vnd.openxmlformats-officedocument.wordprocessingml.document';
        }

        if (strlen($base64Data) > 100) {
            Log::warning("No clear signature detected, defaulting to video/mp4 for base64: " . substr($base64String, 0, 50) . "...");
            return 'video/mp4';
        }

        return 'invalid';
    }

    /**
     * Helper method to validate base64 string
     */
    private function isValidBase64($string)
    {
        $base64Data = preg_replace('/^data:[\w\/]+;base64,/', '', $string);
        if (!preg_match('/^[A-Za-z0-9+\/=]+$/', $base64Data)) {
            return false;
        }
        if (strlen($base64Data) % 4 !== 0) {
            return false;
        }
        return true;
    }

    /**
     * Compress video to ensure it's under 100MB
     */
    private function compressVideo($data, $mimeType, $originalName)
    {
        $inputPath = tempnam(sys_get_temp_dir(), 'video_');
        $outputPath = tempnam(sys_get_temp_dir(), 'compressed_');
        file_put_contents($inputPath, $data);

        $extension = [
            'video/mp4' => 'mp4',
            'video/x-msvideo' => 'avi',
            'video/quicktime' => 'mov'
        ][$mimeType] ?? 'mp4';

        $command = "ffmpeg -i {$inputPath} -vcodec libx264 -crf 35 -preset fast -b:v 300k -vf scale=640:360 -acodec aac -b:a 64k {$outputPath}.{$extension} 2>&1";
        exec($command, $output, $returnCode);

        if ($returnCode !== 0) {
            unlink($inputPath);
            throw new \Exception("Video compression failed for {$originalName}: " . implode("\n", $output));
        }

        $compressedData = file_get_contents("{$outputPath}.{$extension}");
        unlink($inputPath);
        unlink("{$outputPath}.{$extension}");

        if (strlen($compressedData) > 102400 * 1024) {
            throw new \Exception("Compressed video {$originalName} still exceeds 100MB.");
        }

        return $compressedData;
    }

    /**
     * Fetch YouTube video duration using API
     */
    private function getYouTubeDuration($videoUrl)
    {
        if (!$videoUrl) return 5; // Fallback in minutes
        preg_match('/v=([^&]+)/', $videoUrl, $matches);
        $videoId = $matches[1] ?? '';
        $apiKey = env('YOUTUBE_API_KEY', 'AIzaSyDiAiyNJJEnAX7y9zX8zCu-WsPj_f1Z-Jw');
        $url = "https://www.googleapis.com/youtube/v3/videos?part=contentDetails&id={$videoId}&key={$apiKey}";
        $response = @file_get_contents($url);
        if ($response === false) {
            Log::warning("Failed to fetch YouTube duration for video ID: {$videoId}");
            return 5; // Fallback in minutes
        }
        $data = json_decode($response, true);
        if (isset($data['items'][0]['contentDetails']['duration'])) {
            $isoDuration = $data['items'][0]['contentDetails']['duration'];
            return round($this->iso8601ToSeconds($isoDuration) / 60); // Return minutes, rounded
        }
        return 5; // Fallback
    }

    private function iso8601ToSeconds($duration)
    {
        $interval = new \DateInterval($duration);
        return ($interval->d * 86400) + ($interval->h * 3600) + ($interval->i * 60) + $interval->s;
    }
}
