<?php

namespace App\Http\Controllers;

use App\Models\Language;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class LanguageController extends Controller
{
    public function index(): JsonResponse
    {
        $languages = Language::all();
        return response()->json([
            'status' => 'success',
            'data' => $languages,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $language = Language::findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $language,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Code' => 'required|string|max:10|unique:Languages',
            'Name' => 'required|string|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $language = Language::create([
            'Code' => $request->Code,
            'Name' => $request->Name,
            'Created_at' => now(),
        ]);

        return response()->json([
            'status' => 'success',
            'data' => $language,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $language = Language::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Code' => 'sometimes|string|max:10|unique:Languages,Code,' . $id . ',Language_id',
            'Name' => 'sometimes|string|max:100',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $language->update($request->only(['Code', 'Name']));

        return response()->json([
            'status' => 'success',
            'data' => $language,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $language = Language::findOrFail($id);
        $language->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Language deleted successfully',
        ], 200);
    }
}
