<?php

namespace App\Http\Controllers;

use App\Models\InterviewTool;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class InterviewToolController extends Controller
{
    public function index(): JsonResponse
    {
        $tools = InterviewTool::with(['user', 'questionSet'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $tools,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $tool = InterviewTool::with(['user', 'questionSet'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $tool,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:users,User_id',
            'Question_set_id' => 'nullable|exists:exam_questions,Question_id',
            'Tool_type' => 'required|in:mock_interview,practice_test',
            'Title' => 'required|string|max:255',
            'Content_url' => 'nullable|string|max:255',
            'Score' => 'nullable|numeric|between:0,100',
            'Config' => 'nullable|json',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $tool = InterviewTool::create($request->only([
            'User_id',
            'Question_set_id',
            'Tool_type',
            'Title',
            'Content_url',
            'Score',
            'Config',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $tool,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $tool = InterviewTool::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'User_id' => 'sometimes|exists:users,User_id',
            'Question_set_id' => 'sometimes|nullable|exists:exam_questions,Question_id',
            'Tool_type' => 'sometimes|in:mock_interview,practice_test',
            'Title' => 'sometimes|string|max:255',
            'Content_url' => 'nullable|string|max:255',
            'Score' => 'nullable|numeric|between:0,100',
            'Config' => 'nullable|json',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $tool->update($request->only([
            'User_id',
            'Question_set_id',
            'Tool_type',
            'Title',
            'Content_url',
            'Score',
            'Config',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $tool,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $tool = InterviewTool::findOrFail($id);
        $tool->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Interview tool deleted successfully',
        ], 200);
    }
}
?>
