<?php

namespace App\Http\Controllers;

use App\Events\InterviewStatusChanged;
use App\Http\Controllers\CompanyFeatureUsageController;
use App\Models\Interview;
use App\Models\JobApplication;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class InterviewController extends Controller
{
    public function index(): JsonResponse
    {
        $interviews = Interview::with([
            'application' => function ($query) {
                $query->with([
                    'user' => function ($userQuery) {
                        $userQuery->with(['candidateProfile', 'companyProfile']);
                    },
                    'job' => function ($jobQuery) {
                        $jobQuery->with('company');
                    }
                ]);
            }
        ])->get();

        $interviews->each(function ($interview) {
            if ($interview->application && $interview->application->user) {
                if ($interview->application->user->User_type === 'candidate') {
                    $interview->application->user->profile = $interview->application->user->candidateProfile;
                } elseif ($interview->application->user->User_type === 'company') {
                    $interview->application->user->profile = $interview->application->user->companyProfile;
                }
                unset($interview->application->user->candidateProfile);
                unset($interview->application->user->companyProfile);

                $interview->application->user->created_at = $interview->application->user->created_at ? Carbon::parse($interview->application->user->created_at)->format('Y-m-d H:i:s') : null;
                $interview->application->user->updated_at = $interview->application->user->updated_at ? Carbon::parse($interview->application->user->updated_at)->format('Y-m-d H:i:s') : null;

                if ($interview->application->user->profile) {
                    $interview->application->user->profile->Created_at = $interview->application->user->profile->Created_at ? Carbon::parse($interview->application->user->profile->Created_at)->format('Y-m-d H:i:s') : null;
                    $interview->application->user->profile->Updated_at = $interview->application->user->profile->Updated_at ? Carbon::parse($interview->application->user->profile->Updated_at)->format('Y-m-d H:i:s') : null;
                }

                if ($interview->application->job) {
                    $interview->application->job->created_at = $interview->application->job->created_at ? Carbon::parse($interview->application->job->created_at)->format('Y-m-d H:i:s') : null;
                    $interview->application->job->updated_at = $interview->application->job->updated_at ? Carbon::parse($interview->application->job->updated_at)->format('Y-m-d H:i:s') : null;
                    $interview->application->job->deadline = $interview->application->job->deadline ? Carbon::parse($interview->application->job->deadline)->format('Y-m-d H:i:s') : null;
                }
            }

            $interview->Date = $interview->Date ? Carbon::parse($interview->Date)->format('Y-m-d H:i:s') : null;
            $interview->created_at = $interview->created_at ? Carbon::parse($interview->created_at)->format('Y-m-d H:i:s') : null;
            $interview->updated_at = $interview->updated_at ? Carbon::parse($interview->updated_at)->format('Y-m-d H:i:s') : null;
        });

        return response()->json([
            'status' => 'success',
            'data' => $interviews,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $interview = Interview::with([
            'application' => function ($query) {
                $query->with([
                    'user' => function ($userQuery) {
                        $userQuery->with(['candidateProfile', 'companyProfile']);
                    },
                    'job' => function ($jobQuery) {
                        $jobQuery->with('company');
                    }
                ]);
            }
        ])->findOrFail($id);

        if ($interview->application && $interview->application->user) {
            if ($interview->application->user->User_type === 'candidate') {
                $interview->application->user->profile = $interview->application->user->candidateProfile;
            } elseif ($interview->application->user->User_type === 'company') {
                $interview->application->user->profile = $interview->application->user->companyProfile;
            }
            unset($interview->application->user->candidateProfile);
            unset($interview->application->user->companyProfile);

            $interview->application->user->created_at = $interview->application->user->created_at ? Carbon::parse($interview->application->user->created_at)->format('Y-m-d H:i:s') : null;
            $interview->application->user->updated_at = $interview->application->user->updated_at ? Carbon::parse($interview->application->user->updated_at)->format('Y-m-d H:i:s') : null;

            if ($interview->application->user->profile) {
                $interview->application->user->profile->Created_at = $interview->application->user->profile->Created_at ? Carbon::parse($interview->application->user->profile->Created_at)->format('Y-m-d H:i:s') : null;
                $interview->application->user->profile->Updated_at = $interview->application->user->profile->Updated_at ? Carbon::parse($interview->application->user->profile->Updated_at)->format('Y-m-d H:i:s') : null;
            }

            if ($interview->application->job) {
                $interview->application->job->created_at = $interview->application->job->created_at ? Carbon::parse($interview->application->job->created_at)->format('Y-m-d H:i:s') : null;
                $interview->application->job->updated_at = $interview->application->job->updated_at ? Carbon::parse($interview->application->job->updated_at)->format('Y-m-d H:i:s') : null;
                $interview->application->job->deadline = $interview->application->job->deadline ? Carbon::parse($interview->application->job->deadline)->format('Y-m-d H:i:s') : null;
            }
        }

        $interview->Date = $interview->Date ? Carbon::parse($interview->Date)->format('Y-m-d H:i:s') : null;
        $interview->created_at = $interview->created_at ? Carbon::parse($interview->created_at)->format('Y-m-d H:i:s') : null;
        $interview->updated_at = $interview->updated_at ? Carbon::parse($interview->updated_at)->format('Y-m-d H:i:s') : null;

        return response()->json([
            'status' => 'success',
            'data' => $interview,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        Log::info('Accessed InterviewController::store', ['request' => $request->all()]);

        $validator = Validator::make($request->all(), [
            'Application_id' => [
                'required',
                'exists:job_applications,Application_id',
                function ($attribute, $value, $fail) {
                    $application = JobApplication::with('user', 'job.company.user')->find($value);
                    if (!$application) {
                        $fail('The selected application ID does not exist.');
                    } elseif (!$application->user || $application->user->User_type !== 'candidate') {
                        Log::warning('Invalid application for interview: User is not a candidate', [
                            'application_id' => $value,
                            'user_id' => $application->user ? $application->user->User_id : null,
                            'user_type' => $application->user ? $application->user->User_type : null,
                        ]);
                        $fail('The selected application ID must belong to a candidate user.');
                    }
                },
            ],
            'Date' => 'required|date_format:Y-m-d H:i:s|after:now',
            'Video_link' => 'nullable|url|max:255',
            'Status' => 'required|in:scheduled,completed,canceled',
            'Feedback' => 'nullable|string|max:1000',
            'Candidate_rating' => 'nullable|numeric|between:0,99.99',
        ], [
            'Application_id.required' => 'The application ID is required.',
            'Application_id.exists' => 'The selected application ID does not exist.',
            'Date.required' => 'The interview date is required.',
            'Date.date_format' => 'The interview date must be in the format Y-m-d H:i:s.',
            'Date.after' => 'The interview date must be in the future.',
            'Video_link.url' => 'The video link must be a valid URL.',
            'Video_link.max' => 'The video link must not exceed 255 characters.',
            'Status.required' => 'The status is required.',
            'Status.in' => 'The status must be one of: scheduled, completed, canceled.',
            'Feedback.max' => 'The feedback must not exceed 1000 characters.',
            'Candidate_rating.numeric' => 'The candidate rating must be a number.',
            'Candidate_rating.between' => 'The candidate rating must be between 0 and 99.99.',
        ]);

        if ($validator->fails()) {
            Log::error('Interview creation validation failed', [
                'errors' => $validator->errors()->toArray(),
                'request_data' => $request->all(),
            ]);
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $application = JobApplication::with('job.company.user')->findOrFail($request->input('Application_id'));

        $interview = Interview::create($request->only([
            'Application_id',
            'Date',
            'Video_link',
            'Status',
            'Feedback',
            'Candidate_rating',
        ]));

        try {
            event(new InterviewStatusChanged($interview));
            Log::info('InterviewStatusChanged event fired for interview creation', [
                'interview_id' => $interview->Interview_id,
                'application_id' => $interview->Application_id,
                'status' => $interview->Status,
                'candidate_user_id' => $application->User_id,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to fire InterviewStatusChanged event: ' . $e->getMessage(), [
                'interview_id' => $interview->Interview_id,
            ]);
        }

        $interview->Date = $interview->Date ? Carbon::parse($interview->Date)->format('Y-m-d H:i:s') : null;
        $interview->created_at = $interview->created_at ? Carbon::parse($interview->created_at)->format('Y-m-d H:i:s') : null;
        $interview->updated_at = $interview->updated_at ? Carbon::parse($interview->updated_at)->format('Y-m-d H:i:s') : null;

        return response()->json([
            'status' => 'success',
            'data' => $interview,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $interview = Interview::with('application.job.company.user')->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Application_id' => [
                'sometimes',
                'exists:job_applications,Application_id',
                function ($attribute, $value, $fail) {
                    $application = JobApplication::with('user', 'job.company.user')->find($value);
                    if (!$application) {
                        $fail('The selected application ID does not exist.');
                    } elseif (!$application->user || $application->user->User_type !== 'candidate') {
                        Log::warning('Invalid application for interview update: User is not a candidate', [
                            'application_id' => $value,
                            'user_id' => $application->user ? $application->user->User_id : null,
                            'user_type' => $application->user ? $application->user->User_type : null,
                        ]);
                        $fail('The selected application ID must belong to a candidate user.');
                    }
                },
            ],
            'Date' => 'sometimes|date_format:Y-m-d H:i:s',
            'Video_link' => 'nullable|url|max:255',
            'Status' => 'sometimes|in:scheduled,completed,canceled',
            'Feedback' => 'nullable|string|max:1000',
            'Candidate_rating' => 'nullable|numeric|between:0,99.99',
        ], [
            'Application_id.exists' => 'The selected application ID does not exist.',
            'Date.date_format' => 'The interview date must be in the format Y-m-d H:i:s.',
            'Video_link.url' => 'The video link must be a valid URL.',
            'Video_link.max' => 'The video link must not exceed 255 characters.',
            'Status.in' => 'The status must be one of: scheduled, completed, canceled.',
            'Feedback.max' => 'The feedback must not exceed 1000 characters.',
            'Candidate_rating.numeric' => 'The candidate rating must be a number.',
            'Candidate_rating.between' => 'The candidate rating must be between 0 and 99.99.',
        ]);

        if ($validator->fails()) {
            Log::error('Interview update validation failed', [
                'interview_id' => $id,
                'errors' => $validator->errors()->toArray(),
                'request_data' => $request->all(),
            ]);
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $originalStatus = $interview->Status;
        $data = $request->only([
            'Application_id',
            'Date',
            'Video_link',
            'Status',
            'Feedback',
            'Candidate_rating',
        ]);

        $interview->update($data);

        if (isset($data['Status']) && $data['Status'] !== $originalStatus) {
            try {
                event(new InterviewStatusChanged($interview));
                Log::info('InterviewStatusChanged event fired for interview status update', [
                    'interview_id' => $interview->Interview_id,
                    'application_id' => $interview->Application_id,
                    'old_status' => $originalStatus,
                    'new_status' => $interview->Status,
                    'candidate_user_id' => $interview->application->User_id,
                ]);
            } catch (\Exception $e) {
                Log::error('Failed to fire InterviewStatusChanged event: ' . $e->getMessage(), [
                    'interview_id' => $interview->Interview_id,
                ]);
            }
        }

        $interview->Date = $interview->Date ? Carbon::parse($interview->Date)->format('Y-m-d H:i:s') : null;
        $interview->created_at = $interview->created_at ? Carbon::parse($interview->created_at)->format('Y-m-d H:i:s') : null;
        $interview->updated_at = $interview->updated_at ? Carbon::parse($interview->updated_at)->format('Y-m-d H:i:s') : null;

        return response()->json([
            'status' => 'success',
            'data' => $interview,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $interview = Interview::with('application.job.company.user')->findOrFail($id);

        $interview->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Interview deleted successfully',
        ], 200);
    }
}
