<?php

namespace App\Http\Controllers;

use App\Models\InterviewAnswer;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Exception;

class InterviewAnswerController extends Controller
{
    public function index(): JsonResponse
    {
        try {
            $answers = InterviewAnswer::with(['interview', 'question'])->get();
            return response()->json(['success' => true, 'data' => $answers, 'message' => 'Answers retrieved successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to retrieve answers: ' . $e->getMessage()], 500);
        }
    }

    public function store(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'Interview_id' => 'required|exists:interview_attempts,Interview_id',
                'Question_id' => 'required|exists:interview_questions,Interview_question_id',
                'Answer_text' => 'required|string|max:65535',
                'Score' => 'nullable|numeric|between:0,99.99',
                'Feedback' => 'nullable|string|max:65535',
            ]);

            $answer = InterviewAnswer::create($validated);
            $answer->load(['interview', 'question']);
            return response()->json(['success' => true, 'data' => $answer, 'message' => 'Answer created successfully'], 201);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to create answer: ' . $e->getMessage()], 500);
        }
    }

    public function show($id): JsonResponse
    {
        try {
            $answer = InterviewAnswer::with(['interview', 'question'])->findOrFail($id);
            return response()->json(['success' => true, 'data' => $answer, 'message' => 'Answer retrieved successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Answer not found: ' . $e->getMessage()], 404);
        }
    }

    public function update(Request $request, $id): JsonResponse
    {
        try {
            $answer = InterviewAnswer::findOrFail($id);
            $validated = $request->validate([
                'Interview_id' => 'required|exists:interview_attempts,Interview_id',
                'Question_id' => 'required|exists:interview_questions,Interview_question_id',
                'Answer_text' => 'required|string|max:65535',
                'Score' => 'nullable|numeric|between:0,99.99',
                'Feedback' => 'nullable|string|max:65535',
            ]);

            $answer->update($validated);
            $answer->load(['interview', 'question']);
            return response()->json(['success' => true, 'data' => $answer, 'message' => 'Answer updated successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to update answer: ' . $e->getMessage()], 500);
        }
    }

    public function destroy($id): JsonResponse
    {
        try {
            $answer = InterviewAnswer::findOrFail($id);
            $answer->delete();
            return response()->json(['success' => true, 'data' => null, 'message' => 'Answer deleted successfully'], 204);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to delete answer: ' . $e->getMessage()], 500);
        }
    }
}
