<?php

namespace App\Http\Controllers;

use App\Models\ExamQuestion;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class ExamQuestionController extends Controller
{
    public function index(): JsonResponse
    {
        $questions = ExamQuestion::with('course')->get()->map(function ($question) {
            // Ensure Options and Metadata are decoded arrays/objects
            $question->Options = json_decode($question->Options, true);
            $question->Metadata = $question->Metadata ? json_decode($question->Metadata, true) : null;
            return $question;
        });

        return response()->json([
            'status' => 'success',
            'data' => $questions,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $question = ExamQuestion::with('course')->findOrFail($id);
        // Ensure Options and Metadata are decoded arrays/objects
        $question->Options = json_decode($question->Options, true);
        $question->Metadata = $question->Metadata ? json_decode($question->Metadata, true) : null;

        return response()->json([
            'status' => 'success',
            'data' => $question,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Course_id' => 'sometimes|nullable|exists:courses,Course_id',
            'Question_type' => 'sometimes|in:quiz,mock_interview,practice_test',
            'Question_text' => 'required|string',
            'Options' => 'required|array|min:1',
            'Options.*.text' => 'required|string',
            'Options.*.is_correct' => 'required|boolean',
            'Difficulty' => 'sometimes|in:easy,medium,hard',
            'Metadata' => 'sometimes|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Prepare data for creation
        $data = $request->only([
            'Course_id',
            'Question_type',
            'Question_text',
            'Difficulty',
        ]);

        // Convert Options and Metadata to JSON strings for storage
        $data['Options'] = json_encode($request->Options);
        if ($request->has('Metadata')) {
            $data['Metadata'] = json_encode($request->Metadata);
        }

        $question = ExamQuestion::create($data);

        // Ensure Options and Metadata are decoded in the response
        $question->Options = json_decode($question->Options, true);
        $question->Metadata = $question->Metadata ? json_decode($question->Metadata, true) : null;

        return response()->json([
            'status' => 'success',
            'data' => $question,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $question = ExamQuestion::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Course_id' => 'sometimes|nullable|exists:courses,Course_id',
            'Question_type' => 'sometimes|in:quiz,mock_interview,practice_test',
            'Question_text' => 'sometimes|string',
            'Options' => 'sometimes|array|min:1',
            'Options.*.text' => 'required|string',
            'Options.*.is_correct' => 'required|boolean',
            'Difficulty' => 'sometimes|in:easy,medium,hard',
            'Metadata' => 'sometimes|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Prepare data for update
        $data = $request->only([
            'Course_id',
            'Question_type',
            'Question_text',
            'Difficulty',
        ]);

        // Convert Options and Metadata to JSON strings for storage
        if ($request->has('Options')) {
            $data['Options'] = json_encode($request->Options);
        }
        if ($request->has('Metadata')) {
            $data['Metadata'] = json_encode($request->Metadata);
        }

        $question->update($data);

        // Ensure Options and Metadata are decoded in the response
        $question->Options = json_decode($question->Options, true);
        $question->Metadata = $question->Metadata ? json_decode($question->Metadata, true) : null;

        return response()->json([
            'status' => 'success',
            'data' => $question,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $question = ExamQuestion::findOrFail($id);
        $question->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Exam question deleted successfully',
        ], 200);
    }
}
