<?php

namespace App\Http\Controllers;

use App\Models\ExamAttempt;
use App\Models\ExamQuestion;
use App\Models\Certification;
use App\Models\CandidateProfile;
use App\Models\Course;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Mpdf\Mpdf;
use Mpdf\MpdfException;

class ExamAttemptController extends Controller
{
    public function startExam(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:users,User_id',
            'Course_id' => 'required|exists:courses,Course_id',
            'Type' => 'required|in:quiz,final_exam',
            'Interview_id' => 'nullable|exists:interviews,Interview_id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Check for existing incomplete attempt
        $existingAttempt = ExamAttempt::where('User_id', $request->User_id)
            ->where('Course_id', $request->Course_id)
            ->where('Type', $request->Type)
            ->where('Is_completed', false)
            ->first();

        if ($existingAttempt) {
            return response()->json([
                'status' => 'error',
                'message' => 'An incomplete attempt already exists for this course and exam type.',
            ], 403);
        }

        // Check previous attempts to prevent retest if already passed
        $previousAttempts = ExamAttempt::where('User_id', $request->User_id)
            ->where('Course_id', $request->Course_id)
            ->where('Type', $request->Type)
            ->get();

        if ($previousAttempts->contains('Is_passed', true)) {
            return response()->json([
                'status' => 'error',
                'message' => 'You have already passed this exam. No retest allowed.',
            ], 403);
        }

        $course = Course::find($request->Course_id);
        if (!$course) {
            return response()->json([
                'status' => 'error',
                'message' => 'Course not found.',
            ], 404);
        }

        $totalQuestions = ExamQuestion::where('Course_id', $request->Course_id)->count();
        $examQuestionLimit = $course->Exam_Question > 0 ? min($course->Exam_Question, $totalQuestions) : $totalQuestions;

        $attempt = ExamAttempt::create([
            'User_id' => $request->User_id,
            'Course_id' => $request->Course_id,
            'Type' => $request->Type,
            'Interview_id' => $request->Interview_id,
            'Attempted_at' => now(),
            'Pass_threshold' => $course->Pass_threshold ?? 70,
            'Is_completed' => false,
            'Total_questions' => $examQuestionLimit,
        ]);

        return response()->json([
            'status' => 'success',
            'data' => [
                'attempt_id' => $attempt->Attempt_id,
                'message' => 'Exam attempt started successfully.',
            ],
        ], 201);
    }

    public function getUserAttempts($user_id): JsonResponse
    {
        // Validate User_id
        $validator = Validator::make(['User_id' => $user_id], [
            'User_id' => 'required|exists:users,User_id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Fetch all exam attempts for the user with related data
        $attempts = ExamAttempt::with([
            'user.candidateProfile',
            'user.companyProfile.subscription.plan',
            'course',
            'interviewAttempt'
        ])
            ->where('User_id', $user_id)
            ->get();

        // Map attempts to include certificate details for passed quizzes
        $attemptsData = $attempts->map(function ($attempt) {
            $attemptData = $attempt->toArray();

            if ($attempt->Is_passed && $attempt->Type === 'quiz') {
                $certification = Certification::where('User_id', $attempt->User_id)
                    ->where('Course_id', $attempt->Course_id)
                    ->latest('Issue_date')
                    ->first();

                if ($certification) {
                    if (empty($certification->Certificate_url)) {
                        Log::warning('Certificate_url is empty for certification', [
                            'certification_id' => $certification->Certification_id,
                            'user_id' => $attempt->User_id,
                            'course_id' => $attempt->Course_id
                        ]);
                    }

                    $attemptData['certificate'] = [
                        'Certification_id' => $certification->Certification_id,
                        'Certificate_url' => $certification->Certificate_url ? url($certification->Certificate_url) : null,
                        'Issue_date' => $certification->Issue_date,
                        'Expiry_date' => $certification->Expiry_date,
                        'Verification_code' => $certification->Verification_code,
                        'Serial_number' => $certification->Serial_number,
                        'Status' => $certification->Status,
                        'Certificate_version' => $certification->Certificate_version
                    ];
                } else {
                    Log::warning('No certification found for passed quiz attempt', [
                        'attempt_id' => $attempt->Attempt_id,
                        'user_id' => $attempt->User_id,
                        'course_id' => $attempt->Course_id
                    ]);
                }
            }

            return $attemptData;
        });

        return response()->json([
            'status' => 'success',
            'data' => [
                'total_attempts' => $attempts->count(),
                'attempts' => $attemptsData,
            ],
        ], 200);
    }


    public function getExamQuestions($id): JsonResponse
    {
        // Load only the course relationship, not examQuestions
        $attempt = ExamAttempt::with('course')->findOrFail($id);
        $course = $attempt->course;
        $totalQuestions = ExamQuestion::where('Course_id', $attempt->Course_id)->count();
        $examQuestionLimit = $course->Exam_Question > 0 ? min($course->Exam_Question, $totalQuestions) : $totalQuestions;

        if ($attempt->Total_questions !== $examQuestionLimit) {
            $attempt->update(['Total_questions' => $examQuestionLimit]);
        }

        $questions = ExamQuestion::where('Course_id', $attempt->Course_id)
            ->inRandomOrder()
            ->take($examQuestionLimit)
            ->get();

        return response()->json([
            'status' => 'success',
            'data' => [
                'attempt' => $attempt,
                'questions' => $questions->map(function ($question) {
                    $question->Options = json_decode($question->Options, true);
                    $question->Metadata = $question->Metadata ? json_decode($question->Metadata, true) : null;
                    return $question;
                }),
            ],
        ], 200);
    }

    public function submitExam(Request $request, $id): JsonResponse
    {
        $attempt = ExamAttempt::with(['course.examQuestions'])->findOrFail($id);

        $validator = Validator::make($request->all(), [
            'answers' => 'required|array',
            'answers.*.question_id' => 'required|exists:exam_questions,Question_id',
            'answers.*.user_answer' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $course = $attempt->course;
        $totalQuestions = $course->Exam_Question > 0 ? min($course->Exam_Question, ExamQuestion::where('Course_id', $attempt->Course_id)->count()) : ExamQuestion::where('Course_id', $attempt->Course_id)->count();
        $questions = ExamQuestion::where('Course_id', $attempt->Course_id)
            ->whereIn('Question_id', collect($request->answers)->pluck('question_id'))
            ->get();

        $correctAnswers = 0;
        $userAnswers = collect($request->answers)->keyBy('question_id');

        $questionsWithAnswers = $questions->map(function ($question) use ($userAnswers, &$correctAnswers) {
            $userAnswer = $userAnswers->get($question->Question_id);
            $options = json_decode($question->Options, true);
            $correctOption = array_search(true, array_column($options, 'is_correct'));
            $isCorrect = $userAnswer && $options[$correctOption]['text'] === $userAnswer['user_answer'];

            if ($isCorrect) {
                $correctAnswers++;
            }

            return [
                'Question_id' => $question->Question_id,
                'Course_id' => $question->Course_id,
                'Question_text' => $question->Question_text,
                'Difficulty' => $question->Difficulty,
                'Metadata' => $question->Metadata ? json_decode($question->Metadata, true) : null,
                'Options' => $options,
                'user_selected' => $userAnswer ? $userAnswer['user_answer'] : null,
                'is_correct' => $userAnswer ? $isCorrect : null,
                'Created_at' => $question->Created_at,
                'Updated_at' => $question->Updated_at,
            ];
        });

        $score = ($totalQuestions > 0) ? ($correctAnswers / $totalQuestions) * 100 : 0;
        $isPassed = $score >= $attempt->Pass_threshold;

        $attempt->update([
            'Score' => number_format($score, 2),
            'Is_passed' => $isPassed,
            'Is_completed' => true,
            'Total_questions' => $totalQuestions,
        ]);

        $passStatus = $isPassed ? 'pass' : 'fail';
        $attemptData = $attempt->toArray();
        $attemptData['Is_passed'] = $passStatus;

        $responseData = [
            'score' => number_format($score, 2),
            'is_passed' => $passStatus,
            'attempt' => $attemptData,
            'questions' => $questionsWithAnswers,
            'retest_available' => !$isPassed,
        ];

        if ($passStatus === 'pass' && $attempt->Type === 'quiz') {
            $course = Course::find($attempt->Course_id);
            if (!$course) {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Course information unavailable for certification',
                ], 500);
            }

            $certificationData = [
                'User_id' => $attempt->User_id,
                'Course_id' => $course->Course_id,
                'Certificate_url' => null,
                'Issue_date' => now(),
                'Expiry_date' => now()->addYear(),
                'Verification_code' => Str::random(10),
                'Serial_number' => 'CERT-' . Str::random(8),
                'Status' => 'issued',
                'Certificate_version' => 'v1.0',
                'LinkedIn_shared' => false,
            ];

            $certification = Certification::create($certificationData);
            $certification->load(['user.candidateProfile', 'course']);
            try {
                $this->generateCertificatePdf($certification);
                $responseData['certificate'] = [
                    'Certification_id' => $certification->Certification_id,
                    'Certificate_url' => $certification->Certificate_url ? url($certification->Certificate_url) : null,
                    'Issue_date' => $certification->Issue_date,
                    'Expiry_date' => $certification->Expiry_date,
                    'Verification_code' => $certification->Verification_code,
                    'Serial_number' => $certification->Serial_number,
                    'Status' => $certification->Status,
                    'Certificate_version' => $certification->Certificate_version
                ];
                $responseData['message'] = 'Congratulations, you passed! Certificate generated.';
            } catch (\Exception $e) {
                Log::error('Failed to generate certificate PDF: ' . $e->getMessage(), [
                    'certification_id' => $certification->Certification_id,
                    'user_id' => $certification->User_id,
                    'course_id' => $certification->Course_id
                ]);
                $certification->delete(); // Rollback certification creation
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to generate certificate PDF: ' . $e->getMessage(),
                ], 500);
            }
        }

        return response()->json([
            'status' => 'success',
            'data' => $responseData,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        // Fetch the exam attempt with related data
        $attempt = ExamAttempt::with([
            'user.candidateProfile',
            'user.companyProfile.subscription.plan',
            'course',
            'interviewAttempt'
        ])->findOrFail($id);

        $responseData = [
            'attempt' => $attempt
        ];

        // Check if the attempt is passed and is a quiz, then fetch the certificate
        if ($attempt->Is_passed && $attempt->Type === 'quiz') {
            $certification = Certification::where('User_id', $attempt->User_id)
                ->where('Course_id', $attempt->Course_id)
                ->latest('Issue_date')
                ->first();

            if ($certification) {
                // If Certificate_url is null, attempt to regenerate the certificate
                if (empty($certification->Certificate_url)) {
                    Log::warning('Certificate_url is empty, attempting to regenerate certificate', [
                        'certification_id' => $certification->Certification_id,
                        'user_id' => $attempt->User_id,
                        'course_id' => $attempt->Course_id
                    ]);
                    try {
                        $this->generateCertificatePdf($certification);
                    } catch (\Exception $e) {
                        Log::error('Failed to regenerate certificate PDF: ' . $e->getMessage(), [
                            'certification_id' => $certification->Certification_id,
                            'user_id' => $attempt->User_id,
                            'course_id' => $attempt->Course_id
                        ]);
                    }
                }

                $responseData['certificate'] = [
                    'Certification_id' => $certification->Certification_id,
                    'Certificate_url' => $certification->Certificate_url ? url($certification->Certificate_url) : null,
                    'Issue_date' => $certification->Issue_date,
                    'Expiry_date' => $certification->Expiry_date,
                    'Verification_code' => $certification->Verification_code,
                    'Serial_number' => $certification->Serial_number,
                    'Status' => $certification->Status,
                    'Certificate_version' => $certification->Certificate_version
                ];
            } else {
                Log::warning('No certification found for passed quiz attempt', [
                    'attempt_id' => $id,
                    'user_id' => $attempt->User_id,
                    'course_id' => $attempt->Course_id
                ]);
            }
        }

        return response()->json([
            'status' => 'success',
            'data' => $responseData,
        ], 200);
    }

    protected function generateCertificatePdf($certification)
    {
        try {
            $mpdf = new Mpdf();
            $userId = $certification->User_id;
            $candidateProfile = CandidateProfile::where('User_id', $userId)->first();

            if ($candidateProfile && !empty(trim($candidateProfile->First_name)) && !empty(trim($candidateProfile->Last_name))) {
                $fullName = trim($candidateProfile->First_name . ' ' . $candidateProfile->Last_name);
            } else {
                $fullName = 'Unknown Candidate';
                Log::warning('CandidateProfile missing or invalid names for User_id: ' . $userId, [
                    'certification_id' => $certification->Certification_id,
                    'profile_found' => $candidateProfile ? true : false,
                    'first_name' => $candidateProfile ? $candidateProfile->First_name : null,
                    'last_name' => $candidateProfile ? $candidateProfile->Last_name : null
                ]);
            }

            $pdfContent = view('certification', [
                'certification' => $certification,
                'fullName' => $fullName,
                'courseName' => $certification->course ? $certification->course->Title : 'Unknown Course',
            ])->render();

            $mpdf->WriteHTML($pdfContent);
            $fileName = 'cert_' . $certification->Certificate_id . '.pdf';
            $filePath = public_path('storage/CandidateCertificate/' . $fileName);

            // Ensure the directory exists
            $directory = dirname($filePath);
            if (!file_exists($directory)) {
                if (!mkdir($directory, 0755, true)) {
                    throw new \Exception('Failed to create directory: ' . $directory);
                }
            }

            // Verify directory is writable
            if (!is_writable($directory)) {
                throw new \Exception('Directory is not writable: ' . $directory);
            }

            $mpdf->Output($filePath, \Mpdf\Output\Destination::FILE);

            // Verify file was created
            if (!file_exists($filePath)) {
                throw new \Exception('Failed to create certificate file: ' . $filePath);
            }

            $certificateUrl = 'storage/CandidateCertificate/' . $fileName;
            $certification->update([
                'Certificate_url' => $certificateUrl,
            ]);

            // Verify database update
            if (empty($certification->fresh()->Certificate_url)) {
                throw new \Exception('Failed to update Certificate_url in database');
            }
        } catch (MpdfException $e) {
            Log::error('mPDF error generating certificate: ' . $e->getMessage(), [
                'certification_id' => $certification->Certification_id,
                'user_id' => $certification->User_id,
                'course_id' => $certification->Course_id
            ]);
            throw $e;
        } catch (\Exception $e) {
            Log::error('Error generating certificate PDF: ' . $e->getMessage(), [
                'certification_id' => $certification->Certification_id,
                'user_id' => $certification->User_id,
                'course_id' => $certification->Course_id
            ]);
            throw $e;
        }
    }
}
