<?php

namespace App\Http\Controllers;

use App\Models\CourseReview;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Exception;

class CourseReviewController extends Controller
{
    public function index(): JsonResponse
    {
        try {
            $reviews = CourseReview::with(['user', 'course'])->get();
            return response()->json(['success' => true, 'data' => $reviews, 'message' => 'Reviews retrieved successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to retrieve reviews: ' . $e->getMessage()], 500);
        }
    }

    public function store(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'User_id' => 'required|exists:users,User_id',
                'Course_id' => 'required|exists:courses,Course_id',
                'Rating' => 'required|integer|between:1,5',
                'Comment' => 'nullable|string|max:65535',
            ]);

            $review = CourseReview::create($validated);
            $review->load(['user', 'course']);
            return response()->json(['success' => true, 'data' => $review, 'message' => 'Review created successfully'], 201);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to create review: ' . $e->getMessage()], 500);
        }
    }

    public function show($id): JsonResponse
    {
        try {
            $review = CourseReview::with(['user', 'course'])->findOrFail($id);
            return response()->json(['success' => true, 'data' => $review, 'message' => 'Review retrieved successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Review not found: ' . $e->getMessage()], 404);
        }
    }

    public function update(Request $request, $id): JsonResponse
    {
        try {
            $review = CourseReview::findOrFail($id);
            $validated = $request->validate([
                'User_id' => 'required|exists:users,User_id',
                'Course_id' => 'required|exists:courses,Course_id',
                'Rating' => 'required|integer|between:1,5',
                'Comment' => 'nullable|string|max:65535',
            ]);

            $review->update($validated);
            $review->load(['user', 'course']);
            return response()->json(['success' => true, 'data' => $review, 'message' => 'Review updated successfully'], 200);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to update review: ' . $e->getMessage()], 500);
        }
    }

    public function destroy($id): JsonResponse
    {
        try {
            $review = CourseReview::findOrFail($id);
            $review->delete();
            return response()->json(['success' => true, 'data' => null, 'message' => 'Review deleted successfully'], 204);
        } catch (Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to delete review: ' . $e->getMessage()], 500);
        }
    }
}
