<?php

namespace App\Http\Controllers;

use App\Models\Coupon;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class CouponController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(): JsonResponse
    {
        $coupons = Coupon::all();
        return response()->json([
            'status' => 'success',
            'data' => $coupons
        ], 200);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Code' => 'required|string|max:50|unique:coupons,Code',
            'Discount_type' => 'required|in:percentage,fixed',
            'Discount_value' => 'required|numeric|min:0',
            'Expiry_date' => 'required|date|after_or_equal:today',
            'Usage_limit' => 'required|integer|min:0',
            'Used_count' => 'integer|min:0',
            'Status' => 'required|in:active,inactive',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors()
            ], 422);
        }

        
        $coupon = Coupon::create($request->only([
            'Code',
            'Discount_type',
            'Discount_value',
            'Expiry_date',
            'Usage_limit',
            'Used_count',
            'Status',
        ]));
        return response()->json([
            'status' => 'success',
            'data' => $coupon
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(Coupon $coupon): JsonResponse
    {
        return response()->json([
            'status' => 'success',
            'data' => $coupon
        ], 200);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Coupon $coupon): JsonResponse
    {
        // Prepare validation rules
        $rules = [
            'Discount_type' => 'sometimes|required|in:percentage,fixed',
            'Discount_value' => 'sometimes|required|numeric|min:0',
            'Expiry_date' => 'sometimes|required|date|after_or_equal:today',
            'Usage_limit' => 'sometimes|required|integer|min:0',
            'Used_count' => 'sometimes|integer|min:0',
            'Status' => 'sometimes|required|in:active,inactive',
        ];

        // Only validate Code uniqueness if it’s provided and different from the current coupon’s Code
        if ($request->has('Code') && $request->input('Code') !== $coupon->Code) {
            $rules['Code'] = 'required|string|max:50|unique:coupons,Code,' . $coupon->coupon_id . ',coupon_id';
        } else {
            $rules['Code'] = 'sometimes|required|string|max:50';
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors()
            ], 422);
        }

        $coupon->update($request->only([
            'Code',
            'Discount_type',
            'Discount_value',
            'Expiry_date',
            'Usage_limit',
            'Used_count',
            'Status',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $coupon
        ], 200);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Coupon $coupon): JsonResponse
    {
        $coupon->delete();
        return response()->json([
            'status' => 'success',
            'message' => 'Coupon deleted successfully'
        ], 204);
    }
}
