<?php

namespace App\Http\Controllers;

use App\Models\CompanyProfile;
use App\Models\CompanyFeatureUsage;
use App\Models\Plan;
use App\Models\Subscription;
use App\Models\SubscriptionHistory;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Carbon;
use App\Models\Job;
use App\Models\JobApplication;
use App\Models\Interview;
use Illuminate\Support\Facades\File; // For direct filesystem operations
use Illuminate\Support\Facades\Log; // Added to fix Undefined type 'Log'

class CompanyProfileController extends Controller
{
    public function index(): JsonResponse
    {
        $profiles = CompanyProfile::with(['user', 'subscription.plan'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $profiles,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $profile = CompanyProfile::with(['user', 'subscription.plan'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $profile,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:users,User_id',
            'Company_name' => [
                'required',
                'string',
                'max:255',
                \Illuminate\Validation\Rule::unique('company_profiles')->where(function ($query) use ($request) {
                    return $query->where('User_id', $request->User_id);
                }),
            ],
            'Description' => 'nullable|string',
            'Logo_url' => 'nullable|file|mimes:jpg,jpeg,png|max:10240',
            'Location' => 'nullable|string|max:255',
        ], [
            'Logo_url.file' => 'The logo must be a valid file.',
            'Logo_url.mimes' => 'The logo must be a JPG, JPEG, or PNG file.',
            'Logo_url.max' => 'The logo must not exceed 10MB.',
            'Company_name.unique' => 'This company name is already taken for this user.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'User_id',
            'Company_name',
            'Description',
            'Location',
        ]);

        if ($request->hasFile('Logo_url') && $request->file('Logo_url')->isValid()) {
            $logo = $request->file('Logo_url');
            $logoName = time() . '_logo_' . $logo->getClientOriginalName();
            $logoDir = public_path('storage/logos');

            // Ensure the directory exists
            if (!File::exists($logoDir)) {
                File::makeDirectory($logoDir, 0755, true);
                Log::info('Created directory: ' . $logoDir);
            }

            // Move the file directly to public/storage/logos
            $logo->move($logoDir, $logoName);
            $logoPath = public_path('storage/logos/' . $logoName);

            // Verify the file was saved
            if (!File::exists($logoPath)) {
                Log::error('Failed to store logo at: ' . $logoPath);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Failed to store logo file',
                ], 500);
            }

            Log::info('Logo stored at: ' . $logoPath);
            $data['Logo_url'] = url('storage/logos/' . $logoName);
            Log::info('Generated Logo_url: ' . $data['Logo_url']);
        }

        DB::beginTransaction();
        try {
            $profile = CompanyProfile::create($data);

            $basicPlan = Plan::where('Access_Level', 'basic')->where('Status', 'active')->first();
            if ($basicPlan) {
                $subscription = Subscription::create([
                    'Plan_id' => $basicPlan->Plan_id,
                    'Company_id' => $profile->Company_id,
                    'Name' => 'Basic Subscription for ' . $profile->Company_name,
                    'Start_date' => now(),
                    'End_date' => null,
                    'Status' => 'active',
                ]);
                $profile->update(['Subscription_id' => $subscription->Subscription_id]);

                SubscriptionHistory::create([
                    'subscription_id' => $subscription->Subscription_id,
                    'plan_id' => $basicPlan->Plan_id,
                    'company_id' => $profile->Company_id,
                    'name' => $subscription->Name,
                    'start_date' => $subscription->Start_date->toDateTimeString(),
                    'end_date' => null,
                    'status' => $subscription->Status,
                    'amount' => 0,
                    'payment_gateway' => null,
                ]);

                CompanyFeatureUsage::create([
                    'Company_id' => $profile->Company_id,
                    'Plan_id' => $basicPlan->Plan_id,
                    'Subscription_id' => $subscription->Subscription_id,
                    'Feature_Name' => 'job_postings',
                    'Usage_Count' => 0,
                    'Max_Usage' => $basicPlan->Max_Job_Postings,
                    'Start_Date' => now(),
                    'End_Date' => null,
                ]);
                CompanyFeatureUsage::create([
                    'Company_id' => $profile->Company_id,
                    'Plan_id' => $basicPlan->Plan_id,
                    'Subscription_id' => $subscription->Subscription_id,
                    'Feature_Name' => 'interview_schedules',
                    'Usage_Count' => 0,
                    'Max_Usage' => $basicPlan->Max_Interview_Schedules,
                    'Start_Date' => now(),
                    'End_Date' => null,
                ]);
            }

            DB::commit();
            return response()->json([
                'status' => 'success',
                'data' => $profile->load('subscription'),
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Failed to create company profile: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to create company profile',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function update(Request $request, $id): JsonResponse
    {
        $profile = CompanyProfile::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'User_id' => 'sometimes|exists:users,User_id',
            'Company_name' => [
                'sometimes',
                'string',
                'max:255',
                \Illuminate\Validation\Rule::unique('company_profiles')->where(function ($query) use ($request, $profile) {
                    return $query->where('User_id', $request->input('User_id', $profile->User_id))
                        ->where('Company_id', '!=', $profile->Company_id);
                }),
            ],
            'Description' => 'nullable|string',
            'Logo_url' => 'nullable|string',
            'Subscription_id' => 'sometimes|exists:subscriptions,Subscription_id',
        ], [
            'Company_name.unique' => 'This company name is already taken for this user.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'User_id',
            'Company_name',
            'Description',
            'Location',
            'Subscription_id',
        ]);

        if ($request->has('Logo_url') && !empty($request->input('Logo_url'))) {
            $logoInput = $request->input('Logo_url');
            $base64String = null;
            $mimeType = 'image/jpeg';
            if (preg_match('/^data:image\/(jpeg|jpg|png);base64,/', $logoInput)) {
                $base64String = preg_replace('/^data:image\/(jpeg|jpg|png);base64,/', '', $logoInput);
                $mimeType = $this->getMimeTypeFromBase64($logoInput);
            } elseif (preg_match('/^[A-Za-z0-9+\/=]+$/', $logoInput) && $this->isValidBase64($logoInput)) {
                $base64String = $logoInput;
            }

            if ($base64String) {
                if (!$this->isValidBase64($base64String)) {
                    Log::error('Invalid base64 string: ' . $logoInput);
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Invalid logo base64 string',
                    ], 422);
                }
                $logoData = base64_decode($base64String, true);
                if ($logoData === false) {
                    Log::error('Failed to decode base64 string: ' . $logoInput);
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Failed to decode logo base64 data',
                    ], 422);
                }
                if (strlen($logoData) > 10240 * 1024) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'The logo must not exceed 10MB.',
                    ], 422);
                }

                if ($profile->Logo_url) {
                    $oldLogoPath = str_replace(url('storage/'), public_path('storage/'), $profile->Logo_url);
                    if (File::exists($oldLogoPath)) {
                        File::delete($oldLogoPath);
                        Log::info('Old logo deleted: ' . $oldLogoPath);
                    }
                }

                $logoExtension = ['image/jpeg' => 'jpg', 'image/jpg' => 'jpg', 'image/png' => 'png'][$mimeType] ?? 'jpg';
                $logoName = time() . '_logo.' . $logoExtension;
                $logoDir = public_path('storage/logos');
                $logoPath = $logoDir . '/' . $logoName;

                // Ensure the directory exists
                if (!File::exists($logoDir)) {
                    File::makeDirectory($logoDir, 0755, true);
                    Log::info('Created directory: ' . $logoDir);
                }

                // Save the file directly to public/storage/logos
                File::put($logoPath, $logoData);
                if (!File::exists($logoPath)) {
                    Log::error('Failed to store logo at: ' . $logoPath);
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Failed to store logo file',
                    ], 500);
                }

                Log::info('Logo saved to: ' . $logoPath);
                $data['Logo_url'] = url('storage/logos/' . $logoName);
                Log::info('Generated Logo_url: ' . $data['Logo_url']);
            } elseif (filter_var($logoInput, FILTER_VALIDATE_URL) && strpos($logoInput, url('storage/logos/')) === 0) {
                $data['Logo_url'] = $logoInput;
            } else {
                Log::error('Invalid logo input: ' . $logoInput);
                return response()->json([
                    'status' => 'error',
                    'message' => 'Logo must be a valid base64 string or storage URL',
                ], 422);
            }
        }

        $profile->update($data);

        return response()->json([
            'status' => 'success',
            'data' => $profile->load('subscription'),
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $profile = CompanyProfile::findOrFail($id);

        if ($profile->Logo_url) {
            $logoPath = str_replace(url('storage/'), public_path('storage/'), $profile->Logo_url);
            if (File::exists($logoPath)) {
                File::delete($logoPath);
                Log::info('Logo deleted: ' . $logoPath);
            }
        }

        $profile->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Company profile deleted successfully',
        ], 200);
    }

    public function deleteLogo($id): JsonResponse
    {
        $profile = CompanyProfile::findOrFail($id);

        if (!$profile->Logo_url) {
            return response()->json([
                'status' => 'error',
                'message' => 'No logo found in profile.',
            ], 404);
        }

        $logoPath = str_replace(url('storage/'), public_path('storage/'), $profile->Logo_url);
        if (File::exists($logoPath)) {
            File::delete($logoPath);
            Log::info('Logo deleted: ' . $logoPath);
        } else {
            Log::warning('Logo file not found: ' . $logoPath);
        }

        $profile->update(['Logo_url' => null]);

        return response()->json([
            'status' => 'success',
            'message' => 'Logo deleted successfully',
            'data' => $profile,
        ], 200);
    }

    private function getMimeTypeFromBase64($base64String)
    {
        $data = explode(';', $base64String)[0];
        return str_replace('data:', '', $data);
    }

    private function isValidBase64($string)
    {
        if (!preg_match('/^[A-Za-z0-9+\/=]+$/', $string)) {
            return false;
        }
        if (strlen($string) % 4 !== 0) {
            return false;
        }
        return true;
    }

    public function getStats($Company_id): JsonResponse
    {
        $profile = CompanyProfile::findOrFail($Company_id);

        $activeJobs = Job::where('Company_id', $Company_id)
            ->where('Status', 'active')
            ->count();

        $jobIds = Job::where('Company_id', $Company_id)
            ->pluck('Job_id')
            ->toArray();

        $totalApplications = JobApplication::whereIn('Job_id', $jobIds)
            ->count();

        $candidatesHired = JobApplication::whereIn('Job_id', $jobIds)
            ->where('Status', 'accepted')
            ->count();

        $applicationIds = JobApplication::whereIn('Job_id', $jobIds)
            ->pluck('Application_id')
            ->toArray();

        $interviews = Interview::whereIn('Application_id', $applicationIds)
            ->groupBy('Status')
            ->select('Status', DB::raw('count(*) as count'))
            ->pluck('count', 'Status')
            ->toArray();

        $interviewStats = [
            'scheduled' => $interviews['scheduled'] ?? 0,
            'completed' => $interviews['completed'] ?? 0,
            'cancelled' => $interviews['cancelled'] ?? 0,
            'mock' => $interviews['mock'] ?? 0,
        ];

        return response()->json([
            'status' => 'success',
            'data' => [
                'active_jobs' => $activeJobs,
                'total_applications' => $totalApplications,
                'interviews' => $interviewStats,
                'candidates_hired' => $candidatesHired,
            ],
        ], 200);
    }
}
