<?php

namespace App\Http\Controllers;

use App\Models\CompanyFeatureUsage;
use App\Models\CompanyProfile;
use App\Models\Notification;
use App\Models\Plan;
use App\Models\Subscription;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class CompanyFeatureUsageController extends Controller
{
    public function index(): JsonResponse
    {
        $usages = CompanyFeatureUsage::with(['company', 'plan', 'subscription'])->get();
        return response()->json([
            'status' => 'success',
            'data' => $usages,
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $usage = CompanyFeatureUsage::with(['company', 'plan', 'subscription'])->findOrFail($id);
        return response()->json([
            'status' => 'success',
            'data' => $usage,
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Company_id' => 'required|exists:company_profiles,Company_id',
            'Plan_id' => 'nullable|exists:plans,Plan_id',
            'Feature_Name' => 'required|in:job_postings,interview_schedules',
            'Usage_Count' => 'required|integer|min:0',
            'Max_Usage' => 'required|integer|min:0',
            'Start_Date' => 'required|date',
            'End_Date' => 'nullable|date|after_or_equal:Start_Date',
            'Subscription_id' => 'required|exists:subscriptions,Subscription_id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $usage = CompanyFeatureUsage::create($request->only([
            'Company_id',
            'Plan_id',
            'Feature_Name',
            'Usage_Count',
            'Max_Usage',
            'Start_Date',
            'End_Date',
            'Subscription_id',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $usage->load(['company', 'plan', 'subscription']),
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $usage = CompanyFeatureUsage::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Company_id' => 'sometimes|exists:company_profiles,Company_id',
            'Plan_id' => 'sometimes|nullable|exists:plans,Plan_id',
            'Feature_Name' => 'sometimes|in:job_postings,interview_schedules',
            'Usage_Count' => 'sometimes|integer|min:0',
            'Max_Usage' => 'sometimes|integer|min:0',
            'Start_Date' => 'sometimes|date',
            'End_Date' => 'sometimes|nullable|date|after_or_equal:Start_Date',
            'Subscription_id' => 'sometimes|exists:subscriptions,Subscription_id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $usage->update($request->only([
            'Company_id',
            'Plan_id',
            'Feature_Name',
            'Usage_Count',
            'Max_Usage',
            'Start_Date',
            'End_Date',
            'Subscription_id',
        ]));

        return response()->json([
            'status' => 'success',
            'data' => $usage->load(['company', 'plan', 'subscription']),
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $usage = CompanyFeatureUsage::findOrFail($id);
        $usage->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Usage record deleted successfully',
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ], 200);
    }

    public function incrementUsage(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Company_id' => 'required|integer|exists:company_profiles,Company_id',
            'Feature_Name' => 'required|string|in:job_postings,interview_schedules',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $companyId = $request->input('Company_id');
        $featureName = $request->input('Feature_Name');

        $usage = CompanyFeatureUsage::where('Company_id', $companyId)
            ->where('Feature_Name', $featureName)
            ->firstOrFail();

        if ($usage->Usage_Count >= $usage->Max_Usage) {
            return response()->json([
                'status' => 'error',
                'message' => 'Usage limit reached for ' . $usage->Feature_Name,
            ], 400);
        }

        $usage->increment('Usage_Count');

        // Fetch all candidates (users with User_type = 'candidate')
        $candidates = User::where('User_type', 'candidate')->get();
        if ($candidates->isEmpty()) {
            return response()->json([
                'status' => 'error',
                'message' => 'No candidates found to notify',
            ], 400);
        }

        // Create a notification for each candidate
        foreach ($candidates as $candidate) {
            Notification::create([
                'User_id' => $candidate->User_id,
                'Company_id' => $usage->Company_id,
                'Title' => 'New Job Posted',
                'Message' => 'A new job has been posted by Company ID ' . $usage->Company_id . ': ' . $usage->Feature_Name,
                'Type' => 'info', // Changed to 'info' to match allowed values
                'Is_read' => false,
                'Created_at' => now(),
            ]);
        }

        return response()->json([
            'status' => 'success',
            'data' => $usage->load(['company', 'plan', 'subscription']),
            'message' => 'Usage incremented and notifications sent to all candidates',
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ], 200);
    }

    public function getUsageStatus($companyId): JsonResponse
    {
        $usages = CompanyFeatureUsage::where('Company_id', $companyId)->with(['plan', 'subscription'])->get();

        $status = $usages->map(function ($usage) {
            $percentage = ($usage->Max_Usage > 0) ? ($usage->Usage_Count / $usage->Max_Usage) * 100 : 0;
            return [
                'Feature_Name' => $usage->Feature_Name,
                'Usage_Count' => $usage->Usage_Count,
                'Max_Usage' => $usage->Max_Usage,
                'Percentage_Used' => number_format($percentage, 2) . '%',
                'Limit_Reached' => $usage->Usage_Count >= $usage->Max_Usage,
            ];
        });

        return response()->json([
            'status' => 'success',
            'data' => $status,
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ], 200);
    }

    public function getCompanyFeatureUsage($companyId): JsonResponse
    {
        $company = CompanyProfile::find($companyId);
        if (!$company) {
            return response()->json([
                'status' => 'error',
                'message' => 'Company not found',
            ], 404);
        }

        $usages = CompanyFeatureUsage::where('Company_id', $companyId)->with(['plan', 'subscription'])->get();

        if ($usages->isEmpty()) {
            return response()->json([
                'status' => 'success',
                'message' => 'No usage data available for this company',
                'data' => [],
            ], 200);
        }

        $usageData = $usages->map(function ($usage) {
            $percentage = ($usage->Max_Usage > 0) ? ($usage->Usage_Count / $usage->Max_Usage) * 100 : 0;
            return [
                'Feature_Name' => $usage->Feature_Name,
                'Usage_Count' => $usage->Usage_Count,
                'Max_Usage' => $usage->Max_Usage,
                'Percentage_Used' => number_format($percentage, 2) . '%',
                'Limit_Reached' => $usage->Usage_Count >= $usage->Max_Usage,
                'Plan_Name' => $usage->plan ? $usage->plan->Name : 'N/A',
                'Subscription_Status' => $usage->subscription ? $usage->subscription->Status : 'N/A',
                'Last_Updated' => $usage->updated_at ? $usage->updated_at->format('Y-m-d H:i:s') : 'N/A',
            ];
        });

        return response()->json([
            'status' => 'success',
            'data' => $usageData,
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ], 200);
    }

    public function syncUsageLimits($companyId): JsonResponse
    {
        $company = CompanyProfile::findOrFail($companyId);
        $subscription = $company->subscription;
        if (!$subscription) {
            return response()->json([
                'status' => 'error',
                'message' => 'No active subscription found',
            ], 404);
        }

        $plan = $subscription->plan;
        $usages = CompanyFeatureUsage::where('Company_id', $companyId)->get();
        foreach ($usages as $usage) {
            $newMaxUsage = 0;
            if ($usage->Feature_Name === 'job_postings') {
                $newMaxUsage = $usage->Max_Usage + $plan->Max_Job_Postings;
                $usage->update([
                    'Max_Usage' => $newMaxUsage,
                    'Usage_Count' => min($usage->Usage_Count, $newMaxUsage),
                ]);
            } elseif ($usage->Feature_Name === 'interview_schedules') {
                $newMaxUsage = $usage->Max_Usage + $plan->Max_Interview_Schedules;
                $usage->update([
                    'Max_Usage' => $newMaxUsage,
                    'Usage_Count' => min($usage->Usage_Count, $newMaxUsage),
                ]);
            }
        }

        return response()->json([
            'status' => 'success',
            'message' => 'Usage limits synced with current subscription',
            'data' => $usages->load(['plan', 'subscription']),
            'timestamp' => now()->format('Y-m-d H:i:s'),
        ], 200);
    }
}
