<?php

namespace App\Http\Controllers;

use App\Models\CandidateProfile;
use App\Models\UserSkill;
use App\Models\Certification;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;

class CandidateSearchController extends Controller
{
    public function index(Request $request): JsonResponse
    {
        $request->validate([
            'skills' => 'string|nullable',
            'location' => 'string|nullable',
            'min_experience' => 'integer|min:0|nullable',
            'certifications' => 'string|nullable',
        ]);

        $query = CandidateProfile::with(['user', 'user.skills', 'user.certifications.course']);

        if ($request->has('skills')) {
            $skills = array_map('trim', explode(',', $request->input('skills')));
            $query->whereHas('user.skills', function ($q) use ($skills) {
                $q->whereIn('Skill_name', $skills);
            });
        }

        if ($request->has('location')) {
            $query->where('Location', 'LIKE', '%' . $request->input('location') . '%');
        }

        if ($request->has('min_experience')) {
            $query->where('Experience_years', '>=', $request->input('min_experience'));
        }

        if ($request->has('certifications')) {
            $certifications = array_map('trim', explode(',', $request->input('certifications')));
            $query->whereHas('user.certifications.course', function ($q) use ($certifications) {
                $q->whereIn('Course_name', $certifications);
            });
        }

        try {
            $candidates = $query->paginate(10); // Paginate with 10 records per page
            return response()->json([
                'status' => 'success',
                'count' => $candidates->total(),
                'data' => $candidates->items(),
            ], 200);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to search candidates: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Server error: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Display a specific candidate profile.
     *
     * @param int $id
     * @return JsonResponse
     */
    public function show($id): JsonResponse
    {
        try {
            $candidate = CandidateProfile::with(['user', 'user.skills', 'user.certifications.course'])->findOrFail($id);

            return response()->json([
                'status' => 'success',
                'data' => $candidate,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Candidate not found: ' . $e->getMessage(),
            ], 404);
        }
    }

        public function search(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'skills' => 'nullable|array',
            'skills.*' => 'string|max:100',
            'certifications' => 'nullable|array',
            'certifications.*' => 'string|max:255',
            'location' => 'nullable|string|max:255',
            'min_experience' => 'nullable|numeric|min:0',
            'max_experience' => 'nullable|numeric|min:0|gte:min_experience',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        try {
            $query = CandidateProfile::with(['user', 'user.skills', 'user.certifications.course']);

            // Filter by skills
            if ($request->has('skills') && !empty($request->input('skills'))) {
                $skills = $request->input('skills');
                $query->whereHas('user.skills', function ($q) use ($skills) {
                    $q->whereIn('Skill_name', $skills);
                });
            }

            // Filter by certifications
            if ($request->has('certifications') && !empty($request->input('certifications'))) {
                $certifications = $request->input('certifications');
                $query->whereHas('user.certifications.course', function ($q) use ($certifications) {
                    $q->whereIn('Course_name', $certifications);
                });
            }

            // Filter by location
            if ($request->has('location') && !empty($request->input('location'))) {
                $query->where('Location', 'like', '%' . $request->input('location') . '%');
            }

            // Filter by experience range
            if ($request->has('min_experience')) {
                $query->where('Experience_years', '>=', $request->input('min_experience'));
            }
            if ($request->has('max_experience')) {
                $query->where('Experience_years', '<=', $request->input('max_experience'));
            }

            $candidates = $query->get();

            return response()->json([
                'status' => 'success',
                'data' => $candidates,
                'message' => 'Candidates retrieved successfully',
            ], 200);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to search candidates: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to search candidates: ' . $e->getMessage(),
            ], 500);
        }
    }
}
