<?php

namespace App\Http\Controllers;

use App\Models\CandidateProfile;
use App\Models\JobApplication;
use App\Models\UserCourse;
use App\Models\Interview;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class CandidateProfileController extends Controller
{

    public function index(Request $request): JsonResponse
    {
        $request->validate([
            'skills' => 'string|nullable',
            'location' => 'string|nullable',
            'min_experience' => 'integer|min:0|nullable',
            'certifications' => 'string|nullable',
        ]);

        $query = CandidateProfile::with(['user', 'user.skills']);

        if ($request->has('skills')) {
            $skills = array_map('trim', explode(',', $request->input('skills')));
            $query->whereHas('user.skills', function ($q) use ($skills) {
                $q->whereIn('Skill_name', $skills);
            });
        }

        if ($request->has('location')) {
            $query->where('Location', 'LIKE', '%' . $request->input('location') . '%');
        }

        if ($request->has('min_experience')) {
            $query->where('Experience', '>=', $request->input('min_experience'));
        }

        if ($request->has('certifications')) {
            $certifications = array_map('trim', explode(',', $request->input('certifications')));
            $query->where(function ($q) use ($certifications) {
                foreach ($certifications as $cert) {
                    $q->whereJsonContains('Certifications', $cert);
                }
            });
        }

        try {
            $candidates = $query->paginate(10); // Paginate with 10 records per page
            return response()->json([
                'status' => 'success',
                'count' => $candidates->total(),
                'data' => $candidates->items(),
            ], 200);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to search candidates: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Server error: ' . $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Display a specific candidate profile.
     *
     * @param int $id
     * @return JsonResponse
     */
    public function show($id): JsonResponse
    {
        try {
            $candidate = CandidateProfile::with(['user', 'user.skills'])->findOrFail($id);

            return response()->json([
                'status' => 'success',
                'data' => $candidate,
            ], 200);
        } catch (\Exception $e) {
            return response()->json([
                'status' => 'error',
                'message' => 'Candidate not found: ' . $e->getMessage(),
            ], 404);
        }
    }
        public function search(Request $request): JsonResponse
    {
        $request->validate([
            'skills' => 'string|nullable',
            'location' => 'string|nullable',
            'min_experience' => 'integer|min:0|nullable',
            'max_experience' => 'integer|min:0|nullable|gte:min_experience',
        ]);

        try {
            $query = CandidateProfile::with(['user', 'user.skills']);

            // Debug: Log the input parameters
            \Illuminate\Support\Facades\Log::info('Search parameters', $request->all());

            // Filter by skills (single or multiple)
            if ($request->has('skills') && !empty($request->input('skills'))) {
                $skills = array_map('trim', explode(',', $request->input('skills')));
                \Illuminate\Support\Facades\Log::info('Skills filter applied', ['skills' => $skills]);
                $query->whereHas('user.skills', function ($q) use ($skills) {
                    $q->whereIn('Skill_name', $skills);
                });
            }

            // Filter by location (single or multiple, case-insensitive)
            if ($request->has('location') && !empty($request->input('location'))) {
                $locations = array_map('trim', explode(',', $request->input('location')));
                \Illuminate\Support\Facades\Log::info('Location filter applied', ['locations' => $locations]);
                $query->where(function ($q) use ($locations) {
                    foreach ($locations as $location) {
                        $q->orWhereRaw('LOWER(Location) LIKE ?', ['%' . strtolower($location) . '%']);
                    }
                });
            }

            // Filter by experience (single value or range)
            if ($request->has('min_experience')) {
                \Illuminate\Support\Facades\Log::info('Min experience filter applied', ['min_experience' => $request->input('min_experience')]);
                $query->where('Experience', '>=', $request->input('min_experience'));
            }
            if ($request->has('max_experience')) {
                \Illuminate\Support\Facades\Log::info('Max experience filter applied', ['max_experience' => $request->input('max_experience')]);
                $query->where('Experience', '<=', $request->input('max_experience'));
            }

            // Debug: Log the generated SQL query
            \Illuminate\Support\Facades\Log::info('Generated SQL query', [
                'sql' => $query->toSql(),
                'bindings' => $query->getBindings()
            ]);

            $candidates = $query->paginate(10); // Paginate with 10 records per page

            return response()->json([
                'status' => 'success',
                'count' => $candidates->total(),
                'data' => $candidates->items(),
                'message' => 'Candidates retrieved successfully',
            ], 200);
        } catch (\Exception $e) {
            \Illuminate\Support\Facades\Log::error('Failed to search candidates: ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
            return response()->json([
                'status' => 'error',
                'message' => 'Failed to search candidates: ' . $e->getMessage(),
            ], 500);
        }
    }


    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'User_id' => 'required|exists:users,User_id',
            'First_name' => 'nullable|string|max:100',
            'Last_name' => 'nullable|string|max:100',
            'Resume_url' => 'nullable|file|mimes:pdf,doc,docx|max:10240',
            'Certifications' => 'nullable|array',
            'Certifications.*' => 'file|mimes:jpg,jpeg,png,pdf|max:10240',
            'Location' => 'nullable|string|max:255',
            'Experience' => 'nullable|integer|min:0',
        ], [
            'Resume_url.file' => 'The resume must be a valid file.',
            'Resume_url.mimes' => 'The resume must be a PDF, DOC, or DOCX file.',
            'Resume_url.max' => 'The resume must not exceed 10MB.',
            'Certifications.*.file' => 'Each certification must be a valid file.',
            'Certifications.*.mimes' => 'Each certification must be a JPG, JPEG, PNG, or PDF file.',
            'Certifications.*.max' => 'Each certification must not exceed 10MB.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'User_id',
            'First_name',
            'Last_name',
            'Location',
            'Experience',
        ]);

        if ($request->hasFile('Resume_url') && $request->file('Resume_url')->isValid()) {
            $resume = $request->file('Resume_url');
            $resumeName = time() . '_' . $resume->getClientOriginalName();
            $resumePath = $resume->storeAs('resumes', $resumeName, 'public');
            $data['Resume_url'] = url('storage/resumes/' . $resumeName);
        }

        if ($request->hasFile('Certifications')) {
            $certificationUrls = [];
            foreach ($request->file('Certifications') as $certification) {
                if ($certification->isValid()) {
                    $certName = time() . '_' . $certification->getClientOriginalName();
                    $certPath = $certification->storeAs('certifications', $certName, 'public');
                    if ($certPath) {
                        $certUrl = url('storage/certifications/' . $certName);
                        $certificationUrls[] = $certUrl;
                    }
                }
            }
            if (!empty($certificationUrls)) {
                $data['Certifications'] = $certificationUrls;
            }
        }

        $profile = CandidateProfile::create($data);

        return response()->json([
            'status' => 'success',
            'data' => $profile,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $profile = CandidateProfile::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'User_id' => 'sometimes|exists:users,User_id',
            'First_name' => 'nullable|string|max:100',
            'Last_name' => 'nullable|string|max:100',
            'Resume_url' => 'nullable|string',
            'Certifications' => 'nullable|array',
            'Certifications.*' => 'nullable|string',
            'Location' => 'nullable|string|max:255',
            'Experience' => 'nullable|integer|min:0',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'User_id',
            'First_name',
            'Last_name',
            'Location',
            'Experience',
        ]);

        // Handle Resume_url
        if ($request->has('Resume_url') && !empty($request->input('Resume_url'))) {
            $resumeInput = $request->input('Resume_url');
            if (preg_match('/^data:application\/(pdf|msword|vnd\.openxmlformats-officedocument\.wordprocessingml\.document);base64,/', $resumeInput)) {
                // Validate base64 string
                $base64String = preg_replace('/^data:application\/(pdf|msword|vnd\.openxmlformats-officedocument\.wordprocessingml\.document);base64,/', '', $resumeInput);
                if (!$this->isValidBase64($base64String)) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Invalid resume base64 string',
                    ], 422);
                }
                $resumeData = base64_decode($base64String, true);
                if ($resumeData === false) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'Failed to decode resume base64 data',
                    ], 422);
                }
                if (strlen($resumeData) > 10240 * 1024) {
                    return response()->json([
                        'status' => 'error',
                        'message' => 'The resume must not exceed 10MB.',
                    ], 422);
                }

                // Delete old resume if exists
                if ($profile->Resume_url) {
                    $oldResumePath = str_replace(url('storage/'), 'public/', $profile->Resume_url);
                    if (Storage::exists($oldResumePath)) {
                        Storage::delete($oldResumePath);
                    }
                }

                $mimeType = $this->getMimeTypeFromBase64($resumeInput);
                $resumeExtension = [
                    'application/pdf' => 'pdf',
                    'application/msword' => 'doc',
                    'application/vnd.openxmlformats-officedocument.wordprocessingml.document' => 'docx'
                ][$mimeType];
                $resumeName = time() . '_resume.' . $resumeExtension;
                $resumePath = 'public/resumes/' . $resumeName;
                Storage::put($resumePath, $resumeData);
                $data['Resume_url'] = url('storage/resumes/' . $resumeName);
            } elseif (filter_var($resumeInput, FILTER_VALIDATE_URL) && strpos($resumeInput, url('storage/resumes/')) === 0) {
                $data['Resume_url'] = $resumeInput;
            } else {
                return response()->json([
                    'status' => 'error',
                    'message' => 'Resume must be a valid base64 string or storage URL',
                ], 422);
            }
        }

        // Handle Certifications
        if ($request->has('Certifications')) {
            $newCertificationUrls = $profile->Certifications ?? [];
            $inputCertifications = $request->input('Certifications', []);

            foreach ($inputCertifications as $index => $certInput) {
                if (empty($certInput)) {
                    continue;
                }

                $base64String = null;
                $mimeType = 'image/jpeg'; // Default to JPEG for raw base64 strings
                if (preg_match('/^data:(image\/(jpeg|jpg|png)|application\/pdf);base64,/', $certInput)) {
                    // Handle standard data URI
                    $base64String = preg_replace('/^data:(image\/(jpeg|jpg|png)|application\/pdf);base64,/', '', $certInput);
                    $mimeType = $this->getMimeTypeFromBase64($certInput);
                } elseif (preg_match('/^[A-Za-z0-9+\/=]+$/', $certInput) && $this->isValidBase64($certInput)) {
                    // Handle raw base64 string, assume JPEG
                    $base64String = $certInput;
                }

                if ($base64String) {
                    // Validate base64 string
                    if (!$this->isValidBase64($base64String)) {
                        continue;
                    }
                    $certData = base64_decode($base64String, true);
                    if ($certData === false) {
                        continue;
                    }
                    if (strlen($certData) > 10240 * 1024) {
                        continue;
                    }

                    // Delete old certification at this index if exists
                    if (isset($newCertificationUrls[$index])) {
                        $oldCertPath = str_replace(url('storage/'), 'public/', $newCertificationUrls[$index]);
                        if (Storage::exists($oldCertPath)) {
                            Storage::delete($oldCertPath);
                        }
                    }

                    $certExtension = [
                        'image/jpeg' => 'jpg',
                        'image/jpg' => 'jpg',
                        'image/png' => 'png',
                        'application/pdf' => 'pdf'
                    ][$mimeType];
                    $certName = time() . '_cert_' . $index . '.' . $certExtension;
                    $certPath = 'public/certifications/' . $certName;
                    Storage::put($certPath, $certData);
                    $newCertificationUrls[$index] = url('storage/certifications/' . $certName);
                } elseif (filter_var($certInput, FILTER_VALIDATE_URL) && strpos($certInput, url('storage/certifications/')) === 0) {
                    $newCertificationUrls[$index] = $certInput;
                } else {
                    continue;
                }
            }

            ksort($newCertificationUrls);
            $data['Certifications'] = array_values($newCertificationUrls);
        }

        $profile->update($data);

        return response()->json([
            'status' => 'success',
            'data' => $profile,
            'message' => !empty($data['Certifications']) ? 'Profile updated, some certifications may have been skipped due to invalid format' : 'Profile updated successfully',
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $profile = CandidateProfile::findOrFail($id);

        if ($profile->Resume_url) {
            $resumePath = str_replace(url('storage/'), 'public/', $profile->Resume_url);
            Storage::delete($resumePath);
        }

        if ($profile->Certifications) {
            foreach ($profile->Certifications as $certUrl) {
                $certPath = str_replace(url('storage/'), 'public/', $certUrl);
                Storage::delete($certPath);
            }
        }

        $profile->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Candidate profile deleted successfully',
        ], 200);
    }

    public function deleteCertification(Request $request, $id): JsonResponse
    {
        $profile = CandidateProfile::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'certification_url' => 'required_without:certification_index|string|url',
            'certification_index' => 'required_without:certification_url|integer|min:0',
        ], [
            'certification_url.required_without' => 'The certification_url field is required when certification_index is not present.',
            'certification_url.string' => 'The certification_url must be a string.',
            'certification_url.url' => 'The certification_url must be a valid URL.',
            'certification_index.required_without' => 'The certification_index field is required when certification_url is not present.',
            'certification_index.integer' => 'The certification_index must be an integer.',
            'certification_index.min' => 'The certification_index must be at least 0.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $certifications = $profile->Certifications ?? [];
        $certUrl = $request->input('certification_url');
        $certIndex = $request->input('certification_index');

        // Determine which certificate to delete
        if ($certIndex !== null && isset($certifications[$certIndex])) {
            $certUrl = $certifications[$certIndex];
        } elseif ($certUrl && !in_array($certUrl, $certifications)) {
            return response()->json([
                'status' => 'error',
                'message' => 'Certification not found in profile. Provided URL: ' . $certUrl . ', Available URLs: ' . json_encode($certifications, JSON_UNESCAPED_SLASHES),
            ], 404);
        } elseif (!$certUrl && $certIndex === null) {
            return response()->json([
                'status' => 'error',
                'message' => 'Either certification_url or certification_index must be provided.',
            ], 422);
        }

        // Remove the certification from the array
        $certifications = array_filter($certifications, function ($url) use ($certUrl) {
            return $url !== $certUrl;
        });

        // Delete the file from storage
        $certPath = str_replace(url('storage/'), 'public/', $certUrl);
        if (Storage::exists($certPath)) {
            Storage::delete($certPath);
        }

        // Update the profile
        $profile->update([
            'Certifications' => array_values($certifications),
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Certification deleted successfully',
            'data' => $profile,
        ], 200);
    }

    public function deleteResume($id): JsonResponse
    {
        $profile = CandidateProfile::findOrFail($id);

        if (!$profile->Resume_url) {
            return response()->json([
                'status' => 'error',
                'message' => 'No resume found in profile.',
            ], 404);
        }

        // Delete the resume file from storage
        $resumePath = str_replace(url('storage/'), 'public/', $profile->Resume_url);
        if (Storage::exists($resumePath)) {
            Storage::delete($resumePath);
        }

        // Update the profile to remove Resume_url
        $profile->update([
            'Resume_url' => null,
        ]);

        return response()->json([
            'status' => 'success',
            'message' => 'Resume deleted successfully',
            'data' => $profile,
        ], 200);
    }

    /**
     * Helper method to get MIME type from base64 string
     */
    private function getMimeTypeFromBase64($base64String)
    {
        $data = explode(';', $base64String)[0];
        return str_replace('data:', '', $data);
    }

    /**
     * Helper method to validate base64 string
     */
    private function isValidBase64($string)
    {
        // Check if the string is valid base64 (contains only valid characters)
        if (!preg_match('/^[A-Za-z0-9+\/=]+$/', $string)) {
            return false;
        }
        // Check if the string has valid padding
        if (strlen($string) % 4 !== 0) {
            return false;
        }
        return true;
    }

    public function getStats($Candidate_id): JsonResponse
    {
        // Validate Candidate_id exists in CandidateProfile
        $profile = CandidateProfile::findOrFail($Candidate_id);
        $userId = $profile->User_id;

        // Count job applications for User_id from CandidateProfile
        $jobApplications = JobApplication::where('User_id', $userId)->count();

        // Count courses for the associated User_id
        $coursesTotal = UserCourse::where('User_id', $userId)->count();
        $coursesCompleted = UserCourse::where('User_id', $userId)->where('Status', 'completed')->count();
        $coursesPending = UserCourse::where('User_id', $userId)->where('Status', 'pending')->count();

        // Get JobApplication IDs for the User_id
        $applicationIds = JobApplication::where('User_id', $userId)->pluck('Application_id')->toArray();

        // Count interviews by status for the associated JobApplication IDs
        $interviews = Interview::whereIn('Application_id', $applicationIds)
            ->groupBy('Status')
            ->select('Status', DB::raw('count(*) as count'))
            ->pluck('count', 'Status')
            ->toArray();

        // Ensure all interview statuses are included, default to 0
        $interviewStats = [
            'scheduled' => $interviews['scheduled'] ?? 0,
            'completed' => $interviews['completed'] ?? 0,
            'cancelled' => $interviews['cancelled'] ?? 0,
            'mock' => $interviews['mock'] ?? 0,
        ];

        return response()->json([
            'status' => 'success',
            'data' => [
                'job_applications' => $jobApplications,
                'courses' => [
                    'total' => $coursesTotal,
                    'completed' => $coursesCompleted,
                    'pending' => $coursesPending,
                ],
                'interviews' => $interviewStats,
            ],
        ], 200);
    }
}
